package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds.{CI, _}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.builder.ReleaseBuilder
import com.xebialabs.xlrelease.domain.{Release, ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository, TriggerRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.{CiIdService, FolderService, ReleaseService}
import com.xebialabs.xlrelease.triggers.actors.TriggerLifecycle
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.triggers.events.{TriggerCreatedFromAsCodeEvent, TriggerUpdatedFromAsCodeEvent}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.utils.CiHelper.fixUpInternalReferences
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.versioning.ascode.ValidationMessage
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.StringValue
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Service
class TriggerAsCodeService @Autowired()(triggerRepository: TriggerRepository,
                                        releaseRepository: ReleaseRepository,
                                        releaseService: ReleaseService,
                                        folderService: FolderService,
                                        referenceSolver: ReferenceSolver,
                                        triggerLifecycle: TriggerLifecycle[Trigger],
                                        ciIdService: CiIdService,
                                        triggerService: TriggerService,
                                        eventBus: XLReleaseEventBus)
  extends Logging {

  def process(context: ImportContext, trigger: Trigger): ImportResult = {

    logger.debug(s"Processing trigger: ${trigger.toString} with metadata ${context.metadata.toString}")

    // check if the release trigger reference is out-of-scope folder that does not exist
    val releaseFolderOutOfScopeExists = trigger match {
      case releaseTrigger: ReleaseTrigger if releaseTrigger.getReleaseFolder != null && !releaseTrigger.getReleaseFolder.startsWith(Ids.LOCAL_FOLDER) =>
        val folderExists = Try(
          folderService.findByPath(releaseTrigger.getReleaseFolder, releaseTrigger.getReleaseFolder.split(Ids.SEPARATOR).length - 1)
        ).toOption
        folderExists.isDefined
      case _ => true // do nothing
    }

    val parentFolder = context.scope.getFolderId.getOrElse(Ids.SEPARATOR)
    val storedTriggers = triggerRepository.findByFolderId(parentFolder, nestedFolders = false, PageRequest.of(0, Int.MaxValue)).getContent.asScala.toList
    validateStoredTriggers(storedTriggers)
    processTrigger(context, trigger)
    storedTriggers.find(_.getTitle == trigger.getTitle) match {
      case Some(existing) =>
        logger.debug(s"Updating trigger: ${existing.toString}")
        update(context, existing, trigger, releaseFolderOutOfScopeExists)
      case None =>
        logger.debug(s"Creating trigger: ${trigger.toString}")
        create(context, trigger, generateId(parentFolder))
    }
  }

  private def processTrigger(context: ImportContext, trigger: Trigger): Unit = {
    val parentFolder = context.scope.getFolderId.getOrElse(Ids.SEPARATOR)
    val home = context.metadata.get(MetadataFields.HOMEFOLDER.toString)

    referenceSolver.resolveReferences(trigger, context.references, parentFolder)
    updateTriggersProperties(trigger, parentFolder, home)
    validateFolderId(trigger)
    updateVariables(trigger)
    fixUpInternalReferences(trigger)
  }

  private def create(context: ImportContext, trigger: Trigger, triggerId: String): ImportResult = {
    trigger.setId(triggerId)
    val created = triggerRepository.create(trigger)
    val messages = validate(context, trigger)
    ImportResult(List(CI.ids.withCreated(created.getId)),
      Seq(
        () => enableTrigger(created),
        () => eventBus.publish(TriggerCreatedFromAsCodeEvent(created, context.scmData))
      ),
      Map.empty,
      messages
    )
  }

  private def update(context: ImportContext, existing: Trigger, trigger: Trigger, releaseFolderOutOfScopeExists: Boolean): ImportResult = {
    trigger.setId(existing.getId)
    trigger.setCiUid(existing.getCiUid)
    copyInternalProperties(existing, trigger)
    val messages = validate(context, trigger, releaseFolderOutOfScopeExists)
    val updated = triggerRepository.update(trigger)
    ImportResult(List(CI.ids.withUpdated(updated.getId)),
      Seq(
        () => triggerService.refreshTrigger(updated),
        () => enableTrigger(updated),
        () => eventBus.publish(TriggerUpdatedFromAsCodeEvent(updated, context.scmData))
      ),
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, trigger: Trigger, releaseFolderOutOfScopeExists: Boolean = true): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(trigger, context.getFolderInfo(), releaseFolderOutOfScopeExists).toList
      case None => List.empty
    }
  }

  private def validateStoredTriggers(storedTriggers: List[Trigger]): Unit = {
    storedTriggers
      .groupBy(_.getTitle)
      .values
      .filter(_.length > 1)
      .foreach { triggerGroup =>
        val trigger = triggerGroup.head
        throw new AsCodeException(s"More than one trigger with the name [${trigger.getTitle}] is present in the repository. Don't know which one to update")
      }
  }

  private def findTemplate(idOrTitle: String, parentFolder: String): Release = {
    if (Ids.isReleaseId(idOrTitle)) {
      findTemplateById(idOrTitle, parentFolder)
    } else {
      findTemplateByTitle(idOrTitle, parentFolder)
    }
  }

  private def findTemplateById(idOrTitle: String, parentFolder: String): Release = {
    try {
      releaseService.findById(idOrTitle)
    }
    catch {
      case nfe: LogFriendlyNotFoundException => findTemplateByTitle(idOrTitle, parentFolder)
      case e: Exception => throw new AsCodeException(s"Failed to find template [$idOrTitle], error [${e.getMessage}].")
    }
  }

  private def findTemplateByTitle(idOrTitle: String, parentFolder: String): Release = {
    releaseService.findTemplatesByTitle(parentFolder, idOrTitle, 0, 2, 1).asScala.toList match {
      case template :: Nil => template
      case _ :: _ => throw new AsCodeException(s"More than one template with name [$idOrTitle] is present in the folder [$parentFolder]. Please use an unique name.")
      case Nil => throw new AsCodeException(s"There is no template with name [$idOrTitle] under folder [$parentFolder]")
    }
  }

  private def resolveReleaseFolderId(releaseFolderId: String): String = {
    //TODO: optimize via caching to reduce number of duplicate queries
    Try {
      folderService.findByPath(releaseFolderId.stripSuffix("/")).getId
    } match {
      case Success(resolvedReleaseFolderID) => resolvedReleaseFolderID
      case Failure(ex) => throw new AsCodeException(s"Unable to resolve release folder id, ${releaseFolderId} for trigger: failure message ${ex.getMessage}")
    }
  }

  private def setReleaseFolder(trigger: ReleaseTrigger): Unit = {
    var folderId = trigger.getReleaseFolder
    if (folderId != null) {
      if (folderId.startsWith(Ids.LOCAL_FOLDER)) {
        val rootFolderPath = folderService.getPath(trigger.getFolderId.split(Ids.SEPARATOR).take(2).mkString(Ids.SEPARATOR)).drop(1).mkString(Ids.SEPARATOR)
        folderId = trigger.getReleaseFolder.replace(Ids.LOCAL_FOLDER, rootFolderPath + Ids.SEPARATOR)
      }
      val releaseFolder = Try { folderService.findByPath(folderId, folderId.split(Ids.SEPARATOR).length - 1) }.toOption match {
        case Some(folder) =>
          folder.getId
        case None =>
          null
      }
      trigger.setReleaseFolder(releaseFolder)
    }
  }

  private def updateTriggersProperties(trigger: Trigger, parentFolder: String, home: Option[String]): Unit = {
    trigger.setFolderId(parentFolder)
    trigger match {
      case t: ReleaseTrigger =>
        t.setTemplate(findTemplate(t.getTemplate, parentFolder).getId)
        setReleaseFolder(t)
      case t: EventBasedTrigger =>
        t.mappedProperties
          .asScala
          .find(p => p.targetProperty == "template")
          .collect { case p: StringValue => findTemplate(p.value, parentFolder) }
          .map { templateId =>
            val value = new StringValue("templateId")
            value.setValue(templateId.getId)
            t.mappedProperties.add(value)
          }
        t.mappedProperties.removeIf(p => p.targetProperty == "template")

        t.mappedProperties
          .asScala
          .find(p => p.targetProperty == "releaseFolder")
          .collect { case p: StringValue => resolveReleaseFolderId(FolderAsCodeService.absolutePath(p.value, home)) }
          .map { folderId =>
            val value = new StringValue("releaseFolderId")
            value.setValue(folderId)
            t.mappedProperties.add(value)
          }
        t.mappedProperties.removeIf(p => p.targetProperty == "releaseFolder")

      case _ => ???
    }
  }

  private def generateId(parentId: String): String = {
    ciIdService.getUniqueId(Type.valueOf(classOf[Trigger]), parentId)
  }

  private def validateFolderId(trigger: Trigger): Unit = {
    if (FolderId(trigger.getFolderId).absolute.equals(Ids.ROOT_FOLDER_ID)) {
      throw new AsCodeException(s"Failed to create trigger ${trigger.getTitle}, you cannot create trigger on the root folder. The root folder path is not supported.")
    }
  }

  private def updateVariables(trigger: Trigger): Unit = {
    //TODO: optimize via caching to reduce number of duplicate queries
    trigger match {
      case t: ReleaseTrigger =>
        val variables = releaseRepository.findById(t.getTemplate, ResolveOptions.WITH_DECORATORS).getVariables.asScala.filter(v => t.getVariablesByKeys.containsKey(v.getKey)).toList.asJava
        val variableHolderTemplate = ReleaseBuilder.newRelease().withId(t.getTemplate).withVariables(variables).build()
        variableHolderTemplate.setVariableValues(t.getTemplateVariableValues(!_.isPassword).asScala.filter(_._2 != null).asJava)
        variableHolderTemplate.setPasswordVariableValues(t.getTemplatePasswordVariables)
        t.setVariables(variableHolderTemplate.getVariables.asScala.filter(_.getId != null).toList.asJava)
      case _ =>
    }
  }

  private def enableTrigger(trigger: Trigger): Unit = {
    if (trigger.isEnabled) {
      triggerLifecycle.enable(trigger, false)
    } else {
      triggerLifecycle.disable(trigger)
    }
  }

  private def copyInternalProperties(existing: Trigger, updatedTrigger: Trigger): Unit = {
    for (internalProperty <- updatedTrigger.getInternalProperties.asScala) {
      updatedTrigger.setProperty(internalProperty, existing.getProperty(internalProperty))
    }
  }
}
