package com.xebialabs.xlrelease.versioning.ascode.actors

import com.xebialabs.xlrelease.actors.ExceptionTranslateActor
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.connector.JGitConnector
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActor._
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnectorInitializer
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersioningConfigService, FolderVersioningService}
import org.apache.pekko.actor.{Actor, ActorLogging, ActorRef, Status}
import org.apache.pekko.cluster.pubsub.DistributedPubSub
import org.apache.pekko.cluster.pubsub.DistributedPubSubMediator.{Publish, Subscribe, SubscribeAck}
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder

import java.util
import scala.util.{Failure, Success, Try, Using}

object FolderVersioningActor {

  sealed trait FolderVersioningCommand {
    val callerContext: Option[Authentication] = Option(SecurityContextHolder.getContext.getAuthentication)
  }

  case class CreateOrUpdateSettings(config: FolderVersioningSettings) extends FolderVersioningCommand

  case class ClearSecrets(folderId: String) extends FolderVersioningCommand

  case class CleanLocalRepo(folderId: String, clusterWide: Boolean = true) extends FolderVersioningCommand

  case class CleanLocalRepoNodeRequested(folderId: String) extends FolderVersioningCommand

  case class ResetLocalRepo(config: FolderVersioningSettings, clusterWide: Boolean = true) extends FolderVersioningCommand

  case class ResetLocalRepoNodeRequested(config: FolderVersioningSettings) extends FolderVersioningCommand

  case class CreateVersion(folderId: String, version: String, description: String) extends FolderVersioningCommand

  case class FetchChanges(folderId: String) extends FolderVersioningCommand

  case class SyncLocalRepoNodeRequested(config: FolderVersioningSettings) extends FolderVersioningCommand
}

trait FolderVersioningActor extends Actor with ActorLogging with ExceptionTranslateActor {
  protected def replyOrFail[T](call: => T): Unit = sender() ! (Try(call) match {
    case Success(t) if t != null => t
    case Success(_) => Status.Failure(new NullPointerException("Method returned null and this cannot be processed"))
    case Failure(ex) =>
      log.error(ex, "Failed to process folder versioning message")
      Status.Failure(translate(ex))
  })


  override def receive: Receive = {
    case cmd: FolderVersioningCommand => handleFolderVersioningCommand(cmd)
    case cmd => log.error(s"Unhandled command $cmd")
  }

  protected def doHandleFolderVersioningCommand(cmd: FolderVersioningCommand): Unit

  protected def handleFolderVersioningCommand(cmd: FolderVersioningCommand): Unit = {
    withSecurityContext(cmd)
  }

  private def withSecurityContext(cmd: FolderVersioningCommand): Unit = {
    val maybeAuthentication = cmd.callerContext
    log.debug(s"Setting caller as ${maybeAuthentication.map(_.getName)} for command $cmd")
    val newCtxt = SecurityContextHolder.createEmptyContext()
    newCtxt.setAuthentication(maybeAuthentication.orNull)
    SecurityContextHolder.setContext(newCtxt)
    try {
      doHandleFolderVersioningCommand(cmd)
    } finally {
      SecurityContextHolder.clearContext()
    }
  }
}

class NonClusteredFolderVersioningActor(folderVersioningService: FolderVersioningService,
                                        connectorInitializer: AsCodeJGitConnectorInitializer,
                                        folderVersioningConfigService: FolderVersioningConfigService) extends FolderVersioningActor {
  override def doHandleFolderVersioningCommand(cmd: FolderVersioningCommand): Unit = cmd match {
    case CleanLocalRepo(folderId, _) =>
      JGitConnector.deleteRepo(folderId)
    case ResetLocalRepo(config, _) =>
      Using.resource(connectorInitializer.init(config)) { connector =>
        connector.resetLocalRepo()
      }
    case ClearSecrets(folderId) =>
      folderVersioningConfigService.findSettings(folderId).foreach { settings =>
        settings.secrets = new util.HashMap[String, String]()
        folderVersioningConfigService.updateSettingsDirectly(settings)
      }
    case FetchChanges(folderId) => replyOrFail {
      folderVersioningService.fetchChanges(folderId)
    }
    case CreateOrUpdateSettings(config) => replyOrFail {
      folderVersioningService.createOrUpdateSettings(config)
    }
    case CreateVersion(folderId, version, description) => replyOrFail {
      folderVersioningService.createVersion(folderId, version, description)
    }
    case _: CleanLocalRepoNodeRequested => ()
    case _: ResetLocalRepoNodeRequested => ()
    case _: SyncLocalRepoNodeRequested => ()
  }
}

class ClusteredFolderVersioningActor(folderVersioningService: FolderVersioningService,
                                     connectorInitializer: AsCodeJGitConnectorInitializer,
                                     folderVersioningConfigService: FolderVersioningConfigService) extends FolderVersioningActor {
  val topicName = "ClusteredFolderVersioningActor"
  val mediator: ActorRef = DistributedPubSub(context.system).mediator
  mediator ! Subscribe(topicName, self)

  override def receive: Receive = {
    case SubscribeAck(Subscribe(topicName, None, _)) =>
      log.info(s"subscribing to topic '$topicName'")
    case cmd: FolderVersioningCommand => handleFolderVersioningCommand(cmd)
  }

  //noinspection ScalaStyle
  override def doHandleFolderVersioningCommand(cmd: FolderVersioningCommand): Unit = cmd match {
    case CleanLocalRepo(folderId, clusterWide) =>
      if (clusterWide) {
        mediator ! Publish(topicName, CleanLocalRepoNodeRequested(folderId))
      } else {
        JGitConnector.deleteRepo(folderId)
      }
    case CleanLocalRepoNodeRequested(folderId) =>
      JGitConnector.deleteRepo(folderId)
    case ResetLocalRepo(config, clusterWide) =>
      if (clusterWide) {
        mediator ! Publish(topicName, ResetLocalRepoNodeRequested(config))
      } else {
        Using.resource(connectorInitializer.init(config)) { connector =>
          connector.resetLocalRepo()
        }
      }
    case ResetLocalRepoNodeRequested(config) =>
      Using.resource(connectorInitializer.init(config)) { connector =>
        connector.resetLocalRepo()
      }
    case ClearSecrets(folderId) =>
      folderVersioningConfigService.findSettings(folderId).foreach { settings =>
        settings.secrets = new util.HashMap[String, String]()
        folderVersioningConfigService.updateSettingsDirectly(settings)
      }
    case FetchChanges(folderId) => replyOrFail {
      val result = folderVersioningService.fetchChanges(folderId)
      val updatedSettings = folderVersioningService.getSettings(folderId)
      mediator ! Publish(topicName, SyncLocalRepoNodeRequested(updatedSettings))
      result
    }
    case SyncLocalRepoNodeRequested(config) =>
      if (sender() != self) {
        Using.resource(connectorInitializer.init(config)) { connector =>
          connector.pull()
        }
      } else {
        log.debug("Ignoring request to sync local repository as it came from current node")
      }
    case CreateOrUpdateSettings(config) => replyOrFail {
      folderVersioningService.createOrUpdateSettings(config)
    }
    case CreateVersion(folderId, version, description) => replyOrFail {
      val result = folderVersioningService.createVersion(folderId, version, description)
      val updatedConfig = folderVersioningService.getSettings(folderId)
      mediator ! Publish(topicName, SyncLocalRepoNodeRequested(updatedConfig))
      result
    }
  }
}
