package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.{FOLDER_VERSIONING_YAML_FILENAME, getDefinitionsPath}
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs, ScmException}
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService.wrapExceptions
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.{AsCodeJGitConnector, AsCodeJGitConnectorInitializer}
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.VersioningStrategyResolver
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import scala.util.{Failure, Success, Try, Using}

@Component
class FolderVersioningPreviewService @Autowired()(connectorInitializer: AsCodeJGitConnectorInitializer,
                                                  folderVersioningConfigService: FolderVersioningConfigService,
                                                  versioningStyleResolver: VersioningStrategyResolver,
                                                  importsCrawler: ImportCrawler) {

  def generatePreview(folderId: String, version: Option[String], fileName: String = FOLDER_VERSIONING_YAML_FILENAME): String = wrapExceptions {
    val config = getSettings(folderId)
    val finalFileName = if (fileName.isEmpty) FOLDER_VERSIONING_YAML_FILENAME else fileName
    val filesToCheckout = getFilesForCheckout(finalFileName, config)
    val blobs: BinaryFile = version match {
      case Some(ver) =>
        generateVersionPreview(finalFileName, config, filesToCheckout, ver)
      case None =>
        generateUnversionedPreview(folderId, finalFileName, config)
    }
    new String(blobs.getContent(), StandardCharsets.UTF_8)
  }

  private def generateVersionPreview(fileName: String, config: FolderVersioningSettings, filesToCheckout: Array[String], ver: String) = {
    val filenameWithPath = getDefinitionsPath(config.scmPath, fileName)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      Try(scmConnector.checkout(filesToCheckout, ver, reset = false)) match {
        case Success(checkedOut) =>
          if (!fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
            verifyRequestedFile(fileName, config, scmConnector, ver)
          }
          getRequestedFile(checkedOut, filenameWithPath)
        case Failure(ex: ScmException) if ex.getMessage.contains("No definition file found for tag") => throw new NotFoundException(ex.getMessage)
        case Failure(ex) => throw ex
      }
    }
  }

  private def generateUnversionedPreview(folderId: String, fileName: String, settings: FolderVersioningSettings) = {

    // TODO optimize - no need to create an entire folder to generate preview
    val allFiles = versioningStyleResolver.resolve(settings.versioningStyle).generateFolder(folderId, settings).blob.filesToAdd
    val requestedFile = allFiles.filter(file => file.absolutePath.substring(settings.scmPath.length + 1).equals(fileName))

    if (requestedFile.isEmpty) {
      throw new NotFoundException("Invalid file requested")
    } else {
      requestedFile.head
    }
  }

  private def getFilesForCheckout(fileName: String, config: FolderVersioningSettings): Array[String] = {
    val scmPath = config.scmPath
    if (fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
      Array(getDefinitionsPath(scmPath, fileName))
    } else {
      Array(getDefinitionsPath(scmPath, fileName), getDefinitionsPath(scmPath, FOLDER_VERSIONING_YAML_FILENAME))
    }
  }

  def verifyRequestedFile(fileName: String, config: FolderVersioningSettings, scm: AsCodeJGitConnector, version: String): Unit = {
    val imports = importsCrawler.findAllImports(config.scmPath)(scm, version)
    if (!imports.contains(fileName)) {
      throw ScmException("The requested file is not amongst the imports.")
    }
  }

  def getRequestedFile(checkedOut: ScmBlobs, fileName: String): BinaryFile = checkedOut.filesToAdd.find(_.absolutePath.equals(fileName)).get

  def getSettings(folderId: String): FolderVersioningSettings = wrapExceptions {
    folderVersioningConfigService.findSettings(folderId).getOrElse(throw ScmException("No version control settings defined for folder"))
  }

}
