package com.xebialabs.xlrelease.versioning.repository.persistence

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, FolderRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class GitRepositoryInterceptor @Autowired()(configurationRepository: ConfigurationRepository,
                                            configurationPersistence: ConfigurationPersistence,
                                            folderRepository: FolderRepository)
  extends PersistenceInterceptor[BaseConfiguration] {

  configurationRepository.registerPersistenceInterceptor(this)

  private lazy val repoBaseType = Type.valueOf("git.Repository")
  private lazy val versioningType = Type.valueOf(classOf[FolderVersioningSettings])

  override def onDelete(ciId: String): Unit = {
    configurationPersistence.getType(ciId) match {
      case Some(ciType) => {
        if (ciType.instanceOf(repoBaseType)) {
          checkUsedByFolderVersioning(ciId)
        }
      }
      case None =>
    }
  }

  private def getAllFolderVersioning(): Seq[FolderVersioningSettings] = {
    configurationRepository.findAllByType[FolderVersioningSettings](versioningType).asScala.toSeq
  }

  private def checkUsedByFolderVersioning(repoId: String): Unit = {
    val settings = getAllFolderVersioning()
    val configTitle = configurationRepository.findConfigurationTitleById(repoId)
    val folders = settings
      .filter(settings => settings.gitConnection.getId.equals(repoId))
      .map(settings => folderRepository.getTitle(settings.getFolderId).get)
    if (folders.nonEmpty) {
      throw new ItemInUseException(
        s"Connection $configTitle is used in Git versioning configuration of ${folders.size} folder(s): " +
          s"${folders.map("'" + _ + "'").mkString("[", ", ", "]")}"
      )
    }
  }
}
