package com.xebialabs.xlrelease.versioning.ascode.scm.connector

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.api.v1.views.VersionInfo
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.scm.connector._
import com.xebialabs.xlrelease.versioning.ascode.VersioningUtils.getEffectiveTagPrefix
import org.eclipse.jgit.api.{Git, PullResult}
import org.eclipse.jgit.lib.{Constants, Ref, Repository}
import org.eclipse.jgit.revwalk.RevWalk
import org.eclipse.jgit.transport.FetchResult
import org.eclipse.jgit.treewalk.TreeWalk
import org.eclipse.jgit.treewalk.filter._

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Using

class AsCodeJGitConnector(config: FolderVersioningSettings) extends JGitConnector {
  private val gitConnection = config.getGitConnection

  override val getConfigId: String = getName(config.getFolderId)

  override lazy val getConnectionSettings: GitConnectionSettings = new GitConnectionSettings(
    gitConnection.getUrl,
    config.getBranch,
    gitConnection.getAuthenticationMethod.name(),
    gitConnection.getUsername,
    PasswordEncrypter.getInstance().ensureDecrypted(gitConnection.getPassword),
    gitConnection.getDomain,
    gitConnection.getProxyHost,
    gitConnection.getProxyPort,
    gitConnection.getProxyUsername,
    PasswordEncrypter.getInstance().ensureDecrypted(gitConnection.getProxyPassword),
    gitConnection.getProxyDomain,
    gitConnection.getCommitterUsername,
    gitConnection.getCommitterEmail
  )

  // TODO check if we can get rid of this stuff for versioning
  // For thread-local settings for Http (authentication settings)
  private var gitClient: GitClient = _
  try {
    gitClient = new GitClient(getConnectionSettings)
  } catch {
    case t: Throwable =>
      throw ScmException("Unable to initialize git client with current settings", t)
  }

  def resetLocalRepo(): Unit = {
    val git = getOrInitGit()
    gitHardResetToOrigin(git)
  }

  def pull(): PullResult = {
    val git = getOrInitGit()
    gitHardResetToOrigin(git)
    pullWithRecovery(git)
  }

  def listVersions(folderVersioningSettings: Option[FolderVersioningSettings] = None): Seq[VersionInfo] = {
    val git = getOrInitGit(folderVersioningSettings.map(_.getGitConnection.getTitle))
    val repository = git.getRepository
    val refDatabase = repository.getRefDatabase
    val tagRefs = refDatabase.getRefsByPrefix(generatePrefix(getEffectiveTagPrefix(config.getScmPath))).asScala.toSeq

    getCommitInfo(repository, tagRefs)
  }

  def pullAndListVersions(folderVersioningSettings: Option[FolderVersioningSettings] = None): Seq[VersionInfo] = {
    val git = getOrInitGit(folderVersioningSettings.map(_.getGitConnection.getTitle))
    gitHardResetToOrigin(git)
    val result = pullWithRecovery(git, null, folderVersioningSettings.map(_.getGitConnection.getTitle))
    updateRefDatabase(git, result.getFetchResult)
    listVersions(folderVersioningSettings)
  }

  private def updateRefDatabase(git: Git, fetchResult: FetchResult) = {
    val tagPrefix = generatePrefix(getEffectiveTagPrefix(config.getScmPath))
    val fetchedRefs = fetchResult.getAdvertisedRefs.asScala.filter(_.getName.startsWith(tagPrefix)).toSeq.map(_.getName)
    val localRefs = git.getRepository.getRefDatabase.getRefsByPrefix(tagPrefix).asScala.toSeq.map(_.getName)
    val refsToRemove = localRefs.diff(fetchedRefs)
    if (refsToRemove.nonEmpty) {
      git.tagDelete().setTags(refsToRemove: _*).call()
    }
  }

  def getVersion(tagName: String, folderVersioningSettings: Option[FolderVersioningSettings] = None): VersionInfo = {
    val git = getOrInitGit(folderVersioningSettings.map(_.getGitConnection.getTitle))
    val repository = git.getRepository
    val tagRef = findTagRef(repository, tagName)
    getCommitInfo(repository, List(tagRef)).head
  }

  def checkout(filePath: Array[String], tagName: String, reset: Boolean = true): ScmBlobs = {
    if(filePath.isEmpty) throw ScmException("Please check the yaml files. There is some invalid yaml format.")
    val filter = PathFilterGroup.createFromStrings(filePath: _*)
    val scmFiles = checkoutInternal(filter, tagName, reset)

    if (scmFiles.isEmpty) throw ScmException(s"No definition file found for tag [$tagName] and paths [${filePath.mkString(",")}]")
    if (scmFiles.size < filePath.length) throw ScmException("The number of requested files is greater than the number of files found in the repository")

    ScmBlobs(scmFiles.toSeq)
  }

  def checkoutByPathAndFilenameRecursive(path: String, filename: String, tagName: String, reset: Boolean = true): ScmBlobs = {
    val filter = AndTreeFilter.create(PathFilter.create(s"$path/"), PathSuffixFilter.create(filename))
    val scmFiles = checkoutInternal(filter, tagName, reset)

    // no git tree entry name filter available, so filter files that have something prefixing "filename" param
    val filteredScmFiles = scmFiles.filter(file => file.absolutePath.split("/").reverse.head.equals(filename))
    ScmBlobs(filteredScmFiles.toSeq)
  }

  private def checkoutInternal(filter: TreeFilter, tagName: String, reset: Boolean = true) = {
    val git = getOrInitGit()
    if (reset) {
      gitHardResetToOrigin(git)
    }
    val repository = git.getRepository

    val tagRef = findTagRef(repository, tagName)

    Using.resource(new RevWalk(repository)) { revWalk =>
      val commit = revWalk.parseCommit(tagRef.getObjectId)
      val tree = commit.getTree
      Using.resource(new TreeWalk(repository)) { treeWalk =>
        treeWalk.addTree(tree)
        treeWalk.setRecursive(true)
        treeWalk.setFilter(filter)
        val scmFiles = collection.mutable.Buffer[BinaryFile]()
        while (treeWalk.next) {
          val objectId = treeWalk.getObjectId(0)
          val loader = repository.open(objectId)
          val fileContent = loader.getBytes()
          scmFiles.addOne(BinaryFile(treeWalk.getPathString, () => fileContent))
        }
        scmFiles
      }
    }
  }

  private def getCommitInfo(repository: Repository, tagRefs: Seq[Ref]): Seq[VersionInfo] = {
    Using.resource(new RevWalk(repository)) { revWalk =>
      tagRefs.map { tagRef =>
        val commit = revWalk.parseCommit(tagRef.getObjectId)
        val version = new VersionInfo
        version.setName(tagRef.getName.substring(10)) // strip the 'refs/tags/'
        version.setShortMessage(commit.getShortMessage)
        version.setFullMessage(commit.getFullMessage)
        version.setAuthor(commit.getAuthorIdent.getName)
        version.setCommiter(commit.getCommitterIdent.getName)
        version.setCommitHash(commit.getName)
        version.setCommitTime(new Date(commit.getCommitTime * 1000L))
        version
      }
    }
  }

  private def findTagRef(repository: Repository, tagName: String): Ref = {
    val tagRef = repository.findRef(s"${Constants.R_TAGS}$tagName")
    if (tagRef == null) {
      throw ScmException(s"Unable to find tag [$tagName] on branch [${config.getBranch}]")
    }
    tagRef
  }

  private def generatePrefix(effectiveTagPrefix: String): String = {
    if (effectiveTagPrefix.endsWith("/")) {
      s"${Constants.R_TAGS}$effectiveTagPrefix"
    } else {
      //ensure prefix ends with '/' to constrain search to exact prefix match
      s"${Constants.R_TAGS}$effectiveTagPrefix/"
    }
  }
}
