package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition, Spec}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.BaseArtifact
import com.xebialabs.xlrelease.utils.CiHelper
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

case class ArtifactResolver(fileResolver: FileResolver) extends Logging {

  def resolveArtifacts(allDefinitions: Definition*): Unit = {
    allDefinitions.collect {
      case Definition(_, _, _, spec: CiSpec) => resolveArtifacts(spec)
    }
  }

  private def resolveArtifacts(spec: Spec): Unit = {
    spec match {
      case ciSpec: CiSpec =>
        val nestedCis = CiHelper.getNestedCis(ciSpec.cis.asJava).asScala
        nestedCis
          .filter(_.getType.instanceOf(typeOf[BaseArtifact]))
          .foreach(setArtifact(ciSpec))
    }
  }

  private def setArtifact(spec: CiSpec): ConfigurationItem => Unit = {
    case artifact: BaseArtifact if artifact.getFileUri == null =>
      if (spec.files.contains(artifact.getPortableFilename())) {
        setArtifactFile(artifact, spec.files)
      } else {
        AsCodeException.throwCiFieldValidationException(artifact.getId, "fileUri", s"CI `${artifact.getId}` has no file or fileUri set")
      }

    case _ =>
  }

  private def setArtifactFile(artifact: BaseArtifact, files: Map[String, String]): Unit = {
    files.get(artifact.getPortableFilename()).foreach { ref =>
      fileResolver.resolve(ref) match {
        case Some(file) =>
          logger.debug(s"Setting file on `${artifact.getId}` to tempfile `${file.toString}`")
          artifact.setFile(file)
          // once we have a file we have to clear portableFilename as it was changed by processor so we would be a able to find real file
          artifact.setPortableFilename(null)
        case None =>
          AsCodeException.throwCiFieldValidationException(
            artifact.getId, "file", s"File `$ref` referenced by `${artifact.getId}` could not be found in the archive"
          )
      }
    }
  }
}
