package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil._
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.FilePerTypeStrategy.{FILE_PER_CI_TYPE, FILE_PER_CI_TYPE_VALUE}
import org.springframework.stereotype.Component

object FilePerTypeStrategy {
  val FILE_PER_CI_TYPE = "filePerCIType"
  val FILE_PER_CI_TYPE_VALUE = "Multiple files - one for each type"
}

/**
 * All entities of a selected type are assumed to be /  will be in one file. I.e. all templates are in one file, all triggers in a separate (but only one) file
 * etc.
 */
@Component
class FilePerTypeStrategy(permissionsChecker: PermissionChecker,
                          folderService: FolderService,
                          folderAsCodeService: FolderAsCodeService,
                          generateService: GenerateService) extends BaseVersioningStrategy {

  override def name: String = FILE_PER_CI_TYPE

  override def value: String = FILE_PER_CI_TYPE_VALUE

  def performGenerate(scope: FolderSearch, rootPath: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    val ciConfigs = settings.splitAndConvert()
    val excludeIds = Seq(settings.getGitConnection.getId)
    ciConfigs.flatMap(performGenerateInternal(_, scope, rootPath, excludeIds)).toMap
  }

  private def performGenerateInternal(ciConfig: CisConfig, scope: FolderSearch, rootPath: String, excludeIds: Seq[String]): Map[String, List[Definition]] = {
    try {
      val generatorConfig = GeneratorConfig(None, scope, ciConfig, permissionsChecker.hasGlobalPermission(ADMIN), excludeIds)
      val definitions = generateService.generate(generatorConfig)

      Map(getDefinitionsPath(scope.path, resolveFileName(ciConfig)) -> definitions.map(_.adjustMetadata(rootPath)))
    } catch {
      case e: AsCodeException =>
        e.getMessage match {
          case GenerateService.EMPTY_DEFINITION_ERROR => Map()
          case _ => throw ScmException(e.getMessage, e.getCause)
        }
    }
  }

  private def resolveFileName(ciConfig: CisConfig) = {
    ciConfig match {
      case c: CisConfig if c.generateConfigurations => CONNECTIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateDashboards => DASHBOARDS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateNotificationSettings => NOTIFICATIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateDeliveryPatterns => PATTERNS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generatePermissions => PERMISSIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateTemplates => TEMPLATES_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateTriggers => TRIGGERS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateVariables => VARIABLES_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateWorkflows => WORKFLOWS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateApplications => APPLICATIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateEnvironments => ENVIRONMENTS_VERSIONING_YAML_FILENAME
      case _ => ""
    }
  }

  override def getFolderService: FolderService = folderService

  override def getFolderAsCodeService: FolderAsCodeService = folderAsCodeService

  override def getPermissions: PermissionChecker = permissionsChecker

  override def isMultiFile: Boolean = true

  override def order: Integer = 10
}
