package com.xebialabs.xlrelease.plugins.dashboard.builder

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.repository.sql.persistence.ListExtensions

import scala.jdk.CollectionConverters._

object DashboardBuilder {
  def newDashboard: DashboardBuilder = new DashboardBuilder

  def newTile: TileBuilder = TileBuilder.newTile
}

class DashboardBuilder {
  private var id: String = _
  private var title: String = _
  private var templateType: Type = Type.valueOf(classOf[Dashboard])
  private var description: String = _
  private var parentId: String = _
  private var owner: String = _
  private var tiles: List[Tile] = List[Tile]()
  private var roleViewers: List[String] = List.empty
  private var roleEditors: List[String] = List.empty
  private var columns: Int = _
  private var properties: Map[String, Any] = Map.empty
  private var autoRefresh: Boolean = _
  private var autoRefreshInterval: Integer = _

  def withId(id: String): DashboardBuilder = {
    this.id = id
    this
  }

  def withTemplateType(templateType: Type) = {
    this.templateType = templateType
    this
  }

  def withTitle(title: String): DashboardBuilder = {
    this.title = title
    this
  }

  def withDescription(description: String): DashboardBuilder = {
    this.description = description
    this
  }

  def withParentId(parentId: String): DashboardBuilder = {
    this.parentId = parentId
    this
  }

  def withOwner(owner: String): DashboardBuilder = {
    this.owner = owner
    this
  }

  def withTiles(tiles: Tile*): DashboardBuilder = {
    this.tiles = tiles.toList
    this
  }

  def withViewers(viewers: String*): DashboardBuilder = {
    this.roleViewers = viewers.toList
    this
  }

  def withEditors(editors: String*): DashboardBuilder = {
    this.roleEditors = editors.toList
    this
  }

  def withColumns(columns: Integer): DashboardBuilder = {
    this.columns = columns
    this
  }

  def withProperties(properties: Map[String, Any]): DashboardBuilder = {
    this.properties = properties
    this
  }

  def withProperty(name: String, value: Any): DashboardBuilder = {
    this.properties = this.properties + (name -> value)
    this
  }

  def withAutoRefresh(autoRefresh: Boolean): DashboardBuilder = {
    this.autoRefresh = autoRefresh
    this
  }

  def withAutoRefreshInterval(autoRefreshInterVal: Integer): DashboardBuilder = {
    this.autoRefreshInterval = autoRefreshInterVal
    this
  }


  def build: Dashboard = {
    val dashboard: Dashboard = templateType.getDescriptor.newInstance(id)
    dashboard.setTitle(title)
    dashboard.setDescription(description)
    dashboard.setParentId(parentId)
    dashboard.setOwner(owner)
    dashboard.setTiles(tiles.asJavaMutable())
    dashboard.setRoleViewers(roleViewers.asJavaMutable())
    dashboard.setRoleEditors(roleEditors.asJavaMutable())
    if (columns != 0) dashboard.setColumns(columns)
    dashboard.setAutoRefresh(autoRefresh)
    dashboard.setAutoRefreshInterval(autoRefreshInterval)
    if (properties != null) {
      properties.foreach(p => dashboard.setProperty(p._1, p._2))
    }
    dashboard
  }
}

object TileBuilder {
  def newTile: TileBuilder = new TileBuilder
}

class TileBuilder {

  private var id: String = _
  private var `type`: String = _
  private var title: String = _
  private var description: String = _
  private var width: Integer = _
  private var height: Integer = _
  private var row: Integer = _
  private var col: Integer = _
  private var properties: Map[String, Any] = Map.empty
  private var variableMappings: Map[String, String] = Map.empty

  def withId(id: String): TileBuilder = {
    this.id = id
    this
  }

  def withType(`type`: String): TileBuilder = {
    this.`type` = `type`
    this
  }

  def withTitle(title: String): TileBuilder = {
    this.title = title
    this
  }

  def withDescription(description: String): TileBuilder = {
    this.description = description
    this
  }

  def withWidth(width: Integer): TileBuilder = {
    this.width = width
    this
  }

  def withHeight(height: Integer): TileBuilder = {
    this.height = height
    this
  }

  def withRow(row: Integer): TileBuilder = {
    this.row = row
    this
  }

  def withCol(col: Integer): TileBuilder = {
    this.col = col
    this
  }

  def withProperties(properties: Map[String, Any]): TileBuilder = {
    this.properties = properties
    this
  }

  def withProperty(name: String, value: Any): TileBuilder = {
    this.properties = this.properties + (name -> value)
    this
  }

  def withVariableMappings(variableMappings: Map[String, String]): TileBuilder = {
    this.variableMappings = variableMappings
    this
  }

  def withVariableMapping(name: String, variable: String): TileBuilder = {
    this.variableMappings = this.variableMappings + (name -> variable)
    this
  }

  def build: Tile = {
    val tile: Tile = Type.valueOf(`type`).getDescriptor.newInstance(id)
    if (title != null) tile.setTitle(title)
    if (description != null) tile.setDescription(description)
    if (width != null) tile.setWidth(width)
    if (height != null) tile.setHeight(height)
    if (row != null) tile.setRow(row)
    if (col != null) tile.setCol(col)
    if (properties != null) {
      properties.foreach(p => tile.setProperty(p._1, p._2))
    }
    if (variableMappings != null) {
      tile.setVariableMapping(variableMappings.asJava)
    }
    tile
  }
}

