package com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence

import com.xebialabs.xlrelease.applications.management.repository.persistence.ManagedApplicationSchema.MANAGED_APPLICATION
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.HomeTilesSqlBuilder.{STMT_APPLICATIONS_BY_FOLDER, STMT_APPLICATIONS_ORDER, STMT_COUNT_APPLICATIONS, STMT_COUNT_APPLICATIONS_BY_FOLDER, STMT_COUNT_APPLICATIONS_BY_FOLDER_GROUP, STMT_EDITED_RELEASES_BY_USER, STMT_EDITED_TEMPLATES_BY_USER, applicationFolderCondition, workflowUserQuery}
import com.xebialabs.xlrelease.repository.sql.persistence.ActivityLogSchema.ACTIVITY_LOGS
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CATEGORIES, CONFIGURATIONS, FOLDERS, RELEASES, RELEASE_CATEGORY_REFS}

object HomeTilesSqlBuilder {

  private val releaseAlias = "R"
  private val templateAlias = "T"
  private val releaseInnerAlias = "RR"
  private val folderAlias = "F"
  private val activityLogsAlias = "AL"
  private val managedApplicationAlias = "APP"
  private val configurationsAlias = "CONF"
  val lastEditedByUserAlias = "LAST_EDITED_BY_USER"
  val releasesCounterAlias = "RELEASE_COUNT"

  private val STMT_LAST_ACTIVITY_LOG_BY_USER =
    s"""
       |SELECT DISTINCT dal.${ACTIVITY_LOGS.CONTAINER_ID},
       |       MAX(dal.${ACTIVITY_LOGS.EVENT_TIME}) as ${ACTIVITY_LOGS.EVENT_TIME}
       |FROM ${ACTIVITY_LOGS.TABLE} dal WHERE dal.${ACTIVITY_LOGS.USERNAME} = :username
       |GROUP BY dal.${ACTIVITY_LOGS.CONTAINER_ID}
       |""".stripMargin

  val STMT_EDITED_TEMPLATES_BY_USER: String =
    s"""
       |SELECT
       |   $releaseAlias.${RELEASES.RELEASE_ID},
       |   $releaseAlias.${RELEASES.RELEASE_TITLE},
       |   $folderAlias.${FOLDERS.NAME},
       |   $folderAlias.${FOLDERS.FOLDER_ID},
       |   $folderAlias.${FOLDERS.FOLDER_PATH},
       |   MAX($activityLogsAlias.${ACTIVITY_LOGS.EVENT_TIME}) AS $lastEditedByUserAlias,
       |   COUNT($releaseInnerAlias.${RELEASES.ORIGIN_TEMPLATE_ID}) AS $releasesCounterAlias
       |FROM ${RELEASES.TABLE} $releaseAlias
       |   JOIN ($STMT_LAST_ACTIVITY_LOG_BY_USER) $activityLogsAlias ON $releaseAlias.${RELEASES.RELEASE_ID} = $activityLogsAlias.${ACTIVITY_LOGS.CONTAINER_ID}
       |   JOIN ${FOLDERS.TABLE} $folderAlias ON $releaseAlias.${RELEASES.FOLDER_CI_UID} = $folderAlias.${FOLDERS.CI_UID}
       |   LEFT JOIN  ${RELEASES.TABLE} $releaseInnerAlias ON $releaseInnerAlias.${RELEASES.ORIGIN_TEMPLATE_ID} = $releaseAlias.${RELEASES.RELEASE_ID}
       |WHERE
       |   $releaseAlias.${RELEASES.STATUS} = 'template' AND $releaseAlias.${RELEASES.KIND} = 'release'
       |GROUP BY $releaseAlias.${RELEASES.RELEASE_ID},
       |         $releaseAlias.${RELEASES.RELEASE_TITLE},
       |         $folderAlias.${FOLDERS.NAME},
       |         $folderAlias.${FOLDERS.FOLDER_ID},
       |         $folderAlias.${FOLDERS.FOLDER_PATH}
       |ORDER BY $lastEditedByUserAlias DESC
       |""".stripMargin

  private val STMT_EDITED_RELEASES_BY_USER: String =
    s"""
       |SELECT
       |       $releaseAlias.${RELEASES.RELEASE_ID},
       |       $releaseAlias.${RELEASES.RELEASE_TITLE},
       |       $releaseAlias.${RELEASES.STATUS},
       |       $releaseAlias.${RELEASES.START_DATE},
       |       $releaseAlias.${RELEASES.END_DATE},
       |       $folderAlias.${FOLDERS.NAME},
       |       $folderAlias.${FOLDERS.FOLDER_ID},
       |       $folderAlias.${FOLDERS.FOLDER_PATH},
       |        MAX($activityLogsAlias.${ACTIVITY_LOGS.EVENT_TIME}) AS $lastEditedByUserAlias
       |FROM ${RELEASES.TABLE} $releaseAlias
       |         JOIN ($STMT_LAST_ACTIVITY_LOG_BY_USER) $activityLogsAlias ON $releaseAlias.${RELEASES.RELEASE_ID} = $activityLogsAlias.${ACTIVITY_LOGS.CONTAINER_ID}
       |         JOIN ${FOLDERS.TABLE} $folderAlias ON $releaseAlias.${RELEASES.FOLDER_CI_UID} = $folderAlias.${FOLDERS.CI_UID}
       |WHERE
       |    $releaseAlias.${RELEASES.STATUS} <> 'template' AND $releaseAlias.${RELEASES.KIND} = 'release'
       |GROUP BY
       |       $releaseAlias.${RELEASES.RELEASE_ID},
       |       $releaseAlias.${RELEASES.RELEASE_TITLE},
       |       $releaseAlias.${RELEASES.STATUS},
       |       $releaseAlias.${RELEASES.START_DATE},
       |       $releaseAlias.${RELEASES.END_DATE},
       |       $folderAlias.${FOLDERS.NAME},
       |       $folderAlias.${FOLDERS.FOLDER_ID},
       |       $folderAlias.${FOLDERS.FOLDER_PATH}
       |ORDER BY $lastEditedByUserAlias DESC
       |""".stripMargin

  private def workflowUserQuery(status: String, owner: String, order: String): String = {
    s"""
       |SELECT
       |    R.${RELEASES.RELEASE_ID},
       |    R.${RELEASES.RELEASE_TITLE},
       |    R.${RELEASES.STATUS},
       |    R.${RELEASES.START_DATE},
       |    R.${RELEASES.END_DATE},
       |    F.${FOLDERS.NAME},
       |    F.${FOLDERS.FOLDER_ID},
       |    F.${FOLDERS.FOLDER_PATH},
       |    CAT.${CATEGORIES.CATEGORY},
       |    T.${RELEASES.CI_UID}
       |
       |FROM ${RELEASES.TABLE} R
       |    JOIN ${FOLDERS.TABLE} F ON R.${RELEASES.FOLDER_CI_UID} = F.${FOLDERS.CI_UID}
       |    JOIN ${RELEASES.TABLE} T ON R.${RELEASES.ORIGIN_TEMPLATE_ID} = T.${RELEASES.RELEASE_ID}
       |    LEFT JOIN ${RELEASE_CATEGORY_REFS.TABLE} CAT_REF ON CAT_REF.${RELEASE_CATEGORY_REFS.sourceColumn} = T.${RELEASES.CI_UID}
       |    LEFT JOIN ${CATEGORIES.TABLE} CAT ON CAT.${CATEGORIES.CI_UID} = CAT_REF.${RELEASE_CATEGORY_REFS.targetColumn}
       |WHERE
       |    $status AND R.${RELEASES.KIND} = 'workflow'
       |    $owner
       |GROUP BY
       |    R.${RELEASES.RELEASE_ID},
       |    R.${RELEASES.RELEASE_TITLE},
       |    R.${RELEASES.STATUS},
       |    R.${RELEASES.START_DATE},
       |    R.${RELEASES.END_DATE},
       |    F.${FOLDERS.NAME},
       |    F.${FOLDERS.FOLDER_ID},
       |    F.${FOLDERS.FOLDER_PATH},
       |    CAT.${CATEGORIES.CATEGORY},
       |    T.${RELEASES.CI_UID}
       |    $order
       |""".stripMargin
  }

  private val STMT_APPLICATIONS_BY_FOLDER =
    s"""
       |SELECT $managedApplicationAlias.${MANAGED_APPLICATION.ID},
       |       $managedApplicationAlias.${MANAGED_APPLICATION.APPLICATION_NAME},
       |       $managedApplicationAlias.${MANAGED_APPLICATION.ENVIRONMENT_TAG},
       |       $managedApplicationAlias.${MANAGED_APPLICATION.DATE_CREATED},
       |       $folderAlias.${FOLDERS.NAME},
       |       $folderAlias.${FOLDERS.FOLDER_ID},
       |       $folderAlias.${FOLDERS.FOLDER_PATH}
       |FROM ${MANAGED_APPLICATION.TABLE} $managedApplicationAlias
       |         JOIN ${CONFIGURATIONS.TABLE} $configurationsAlias ON $managedApplicationAlias.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID} = $configurationsAlias.${CONFIGURATIONS.CI_UID}
       |         JOIN ${FOLDERS.TABLE} $folderAlias ON $configurationsAlias.${CONFIGURATIONS.FOLDER_CI_UID} = $folderAlias.${FOLDERS.CI_UID}
       |""".stripMargin

  private val STMT_APPLICATIONS_ORDER =
    s"""
       | ORDER BY $managedApplicationAlias.${MANAGED_APPLICATION.DATE_CREATED} DESC
       |""".stripMargin

  private def applicationFolderCondition(folders: Seq[Integer]): String = {
    val folderCondition = folders
      .grouped(2)
      .map(pagedIds => s"$folderAlias.${FOLDERS.CI_UID} IN (${pagedIds.map(_.toString).mkString(", ")})")
      .mkString("(", " OR \n", ")")
    folderCondition
  }

  private val STMT_COUNT_APPLICATIONS =
    s"""SELECT COUNT(1)
       | FROM ${MANAGED_APPLICATION.TABLE}
       |""".stripMargin

  private val STMT_COUNT_APPLICATIONS_BY_FOLDER =
    s"""
       |SELECT COUNT($managedApplicationAlias.${MANAGED_APPLICATION.ID}),
       |       $folderAlias.${FOLDERS.NAME},
       |       $folderAlias.${FOLDERS.FOLDER_ID},
       |       $folderAlias.${FOLDERS.FOLDER_PATH}
       |FROM ${MANAGED_APPLICATION.TABLE} $managedApplicationAlias
       |         JOIN ${CONFIGURATIONS.TABLE} $configurationsAlias ON $managedApplicationAlias.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID} = $configurationsAlias.${CONFIGURATIONS.CI_UID}
       |         JOIN ${FOLDERS.TABLE} $folderAlias ON $configurationsAlias.${CONFIGURATIONS.FOLDER_CI_UID} = $folderAlias.${FOLDERS.CI_UID}
       |
       |""".stripMargin

  private val STMT_COUNT_APPLICATIONS_BY_FOLDER_GROUP =
    s"""
       | GROUP BY $folderAlias.${FOLDERS.NAME}, $folderAlias.${FOLDERS.FOLDER_ID}, $folderAlias.${FOLDERS.FOLDER_PATH}
       |""".stripMargin

}

class HomeTilesSqlBuilder(implicit val dialect: Dialect) extends LimitOffset {

  def lastTemplateEditedQuery(limit: Long): String = {
    addLimitAndOffset(STMT_EDITED_TEMPLATES_BY_USER, Some(limit))
  }

  def lastReleaseEditedQuery(limit: Long): String = {
    addLimitAndOffset(STMT_EDITED_RELEASES_BY_USER, Some(limit))
  }

  def lastCreatedApplications(limit: Long, folders: Seq[Integer]): String = {
    val statement = if (folders.isEmpty) {
      throw new IllegalArgumentException("You can't query without folders, something is wrong")
    } else {
      STMT_APPLICATIONS_BY_FOLDER ++ " AND " ++ applicationFolderCondition(folders) ++ STMT_APPLICATIONS_ORDER
    }
    addLimitAndOffset(statement, Some(limit))
  }

  def countApplications(): String = {
    STMT_COUNT_APPLICATIONS
  }

  def countApplicationsByFolder(folders: Seq[Integer]): String = {
    if (folders.isEmpty) {
      throw new IllegalArgumentException("You can't query without folders, something is wrong")
    } else {
      STMT_COUNT_APPLICATIONS_BY_FOLDER ++ " AND " ++ applicationFolderCondition(folders) ++ STMT_COUNT_APPLICATIONS_BY_FOLDER_GROUP
    }
  }

  def lastWorkflowRunning(limit: Long): String = {
    addLimitAndOffset(workflowUserQuery(
      s"R.${RELEASES.STATUS} NOT IN ('completed', 'aborted', 'template')",
      s"AND R.${RELEASES.RELEASE_OWNER} = :owner",
      s"ORDER BY R.${RELEASES.START_DATE}"
    ), Some(limit))
  }

  def lastWorkflowRecent(limit: Long): String = {
    addLimitAndOffset(workflowUserQuery(
      s"R.${RELEASES.STATUS} IN ('completed', 'aborted')",
      s"AND R.${RELEASES.RELEASE_OWNER} = :owner",
      s"ORDER BY R.${RELEASES.END_DATE} DESC"
    ), Some(limit))
  }

}
