import React from 'react';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { XlReactWidgetTreeSelect } from 'xl-react-components';
import moment from 'moment';
import { DotDialog, DotIcon, DotPill, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { primaryButton, secondaryButton } from '../../CommonCardTile/tile-buttons-row.mock';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { DATE_FORMAT_DAY_FIRST, DATE_FORMAT_MONTH_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { ActiveApplicationTile, ActiveApplicationTileProps, ApplicationTileData } from './active-application-tile.component';
import { applicationTile, initialState as applicationTileInitialState } from './ducks/application-tile.reducer';
import colors from '../../../../../../../../../../core/xlr-ui/app/js/colors';

const { setApplicationDialogOpen } = applicationTile.actions;

describe('ActiveApplicationTile', () => {
    const dispatch = jest.fn();

    const title = 'Applications';
    const activeTileData: ApplicationTileData = {
        applications: [
            {
                appCreateDate: 1690187050870,
                appEnvironment: 'test environment',
                appId: 'appId',
                appName: 'Test App',
                folderId: 'folderId',
                folderName: 'test',
                folderPath: '/',
            },
        ],
        folders: [
            {
                folderId: 'folderId',
                folderName: 'test',
                folderPath: '/',
                total: 1,
            },
        ],
        total: 1,
    };
    const headerEndContent = <div>This is end content</div>;

    const defaultProps: ActiveApplicationTileProps = {
        activeTileData,
        headerEndContent,
        primaryButton,
        secondaryButton,
        title,
    };

    const defaultState = {
        applicationTile: { ...applicationTileInitialState, isLoading: false },
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: ActiveApplicationTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ActiveApplicationTile {...props} />, dispatch, state);
    };

    const getCommonActiveTile = (wrapper: ReactWrapper) => wrapper.find(CommonActiveTile);
    const getTable = (wrapper: ReactWrapper) => wrapper.find(DotTable);
    const getRowTitleData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-application-title');
    const getRowFolderData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'application-folder-name');
    const getCreatedDateTooltip = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTooltip) && node.props()['data-testid'] === 'created-date-tooltip');
    const getEnvironmentPill = (wrapper: ReactWrapper) => wrapper.find(DotPill);
    const getDialog = (wrapper: ReactWrapper) => wrapper.find(DotDialog);
    const getDialogDescription = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'application-dialog-description');
    const getDialogInputLabel = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'persistent-label');

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render with correct props', () => {
        const wrapper = mount();

        const commonActiveTileProps = getCommonActiveTile(wrapper).props();
        expect(commonActiveTileProps.avatarIcon).toBe('applications');
        expect(commonActiveTileProps.primaryButton).toBe(primaryButton);
        expect(commonActiveTileProps.secondaryButton).toBe(secondaryButton);
        expect(commonActiveTileProps.subtitle).toBe('Last added - Most recent 5');
        expect(commonActiveTileProps.title).toBe(title);
    });

    it('should render main content correctly', () => {
        const wrapper = mount();

        const columns = [
            { id: 'applications', label: 'Application', truncate: true },
            { id: 'environment', label: 'Environment', width: '136px' },
            { id: 'createdDate', label: 'Created date', width: '136px' },
        ];

        const tableProps = getTable(wrapper).props();
        expect(tableProps.className).toBe('active-application-table');
        expect(tableProps.columns).toStrictEqual(columns);
        expect(tableProps.sortable).toBe(false);
    });

    it('should have correct table data', () => {
        const wrapper = mount();

        const rowTitle = getRowTitleData(wrapper).at(0);
        expect(rowTitle.props().variant).toBe('body1');
        expect(rowTitle).toHaveText(activeTileData.applications[0].appName);

        const rowFolderName = getRowFolderData(wrapper).at(0);
        expect(rowFolderName.props().variant).toBe('body2');
        expect(rowFolderName).toHaveText(activeTileData.applications[0].folderName);

        const environmentPill = getEnvironmentPill(wrapper).at(0);
        const environmentPillProps = environmentPill.props();
        expect(environmentPillProps.backgroundcolor).toBe(colors.successBackground);
        expect(environmentPillProps.bordercolor).toBe(colors.green);
        expect(environmentPillProps.label).toBe(activeTileData.applications[0].appEnvironment);
        expect(environmentPillProps.size).toBe('small');
        expect(environmentPillProps.variant).toBe('outlined');
        const environmentPillIconProps = environmentPill.find(DotIcon).props();
        expect(environmentPillIconProps.className).toContain('environment-infrastructure-icon');
        expect(environmentPillIconProps.iconId).toBe('infrastructure');

        const createdDateTooltip = getCreatedDateTooltip(wrapper);
        const createdDateTooltipProps = createdDateTooltip.props();
        expect(createdDateTooltipProps.title).toBe(moment(activeTileData.applications[0].appCreateDate).format(calculateDotDateFormat(DATE_FORMAT_DAY_FIRST)));
        expect(createdDateTooltipProps.placement).toBe('bottom-start');
        const createdDate = createdDateTooltip.find(DotTypography);
        expect(createdDate.props().variant).toBe('body1');
        expect(createdDate).toHaveText(moment(activeTileData.applications[0].appCreateDate).fromNow());
    });

    it('should display correct date format based on profile settings', () => {
        const wrapper = mount(defaultProps, { ...defaultState, profile: { ...defaultState.profile, dateFormat: DATE_FORMAT_MONTH_FIRST } });
        expect(getCreatedDateTooltip(wrapper).props().title).toBe(
            moment(activeTileData.applications[0].appCreateDate).format(calculateDotDateFormat(DATE_FORMAT_MONTH_FIRST)),
        );
    });

    it('should display dialog', () => {
        const wrapper = mount(defaultProps, { ...defaultState, applicationTile: { ...defaultState.applicationTile, applicationDialogOpen: true } });
        const dialog = getDialog(wrapper);
        const dialogProps = dialog.props();
        expect(dialogProps.cancelButtonProps).toStrictEqual({ label: 'Cancel' });
        expect(dialogProps.className).toBe('card-folder-dialog');
        expect(dialogProps.closeIconVisible).toBe(true);
        expect(dialogProps.closeOnClickAway).toBe(true);
        expect(dialogProps.closeOnSubmit).toBe(false);
        expect(dialogProps.open).toBe(true);
        expect(dialogProps.submitButtonProps).toStrictEqual({ label: 'View applications', disabled: true });
        expect(dialogProps.title).toBe('Choose folder');
        expect(getDialogDescription(wrapper)).toHaveText('Select a folder to show Applications in this folder.');
        expect(getDialogInputLabel(wrapper)).toHaveText('Folder name');
        expect(dialog.find('.xl-react-widget-tree-select-wrapper')).toExist();
    });

    it('should close dialog', () => {
        const wrapper = mount(defaultProps, { ...defaultState, applicationTile: { ...defaultState.applicationTile, applicationDialogOpen: true } });
        getDialog(wrapper).invoke('onCancel')?.('' as never);
        expect(dispatch).toBeCalledWith(setApplicationDialogOpen(false));
    });

    it('should handle on submit dialog', () => {
        const wrapper = mount(defaultProps, { ...defaultState, applicationTile: { ...defaultState.applicationTile, applicationDialogOpen: true } });
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        wrapper.find(XlReactWidgetTreeSelect).invoke('onModelChange')?.({ selectedId: 'folderId' });
        getDialog(wrapper).invoke('onSubmit')?.('' as never);
        expect(window.location.href).toStrictEqual(`http://localhost/#/folders/folderId/application-pipelines/applications-management`);
    });
});
