import React, { useState } from 'react';
import isNil from 'lodash/isNil';
import {
    DotCard,
    DotCardHeader,
    DotAvatar,
    DotCardContent,
    DotTypography,
    DotLink,
    DotCardFooter,
    DotButton,
    DotPill,
    DotIcon,
    DotDialog,
} from '@digital-ai/dot-components';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types
import parseGitUrl from 'parse-github-url';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { XlReactWidgetTreeSelect } from 'xl-react-components';
import { Workflow } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getWorkflowTileState, workflowTile } from '../../ducks/workflow-tile.reducer';
import './workflow-card.component.less';

const { runWorkflow } = workflowTile.actions;

interface WidgetFolder {
    selectedId: string;
    title: string;
}

export const WorkflowCard = (workflow: Workflow) => {
    const { title, author, description, scmTraceabilityData, logo, folderTitle, folderId } = workflow;
    const [isDialogOpen, setDialogOpen] = useState(false);
    const [selectedFolderId, setSelectedFolderId] = useState<string | undefined>(folderId);
    const { folders } = useAppSelector(getWorkflowTileState);
    const dispatch = useAppDispatch();
    const handleOnRun = () => {
        if (!selectedFolderId) return;
        handleOnCancel();
        dispatch(runWorkflow({ workflow, folderId: selectedFolderId }));
    };

    const handleSelectFolder = (folder: WidgetFolder) => {
        setSelectedFolderId(folder.selectedId);
    };

    const handleOnCancel = () => {
        setDialogOpen(false);
        setSelectedFolderId(folderId);
    };

    const cardLogo = logo ? (
        <DotAvatar alt={title} imageSrc={`api/v1/templates/logo/${logo.id}`} type="image" />
    ) : (
        <DotAvatar alt={title} text={''} type="text" />
    );

    const renderCommitLink = () => {
        if (!scmTraceabilityData.commit) return;

        const buildUrlForGithub = () => {
            // looks like we are only compatible with github links? code from scm-commit-data-modal-component.js
            if (!scmTraceabilityData.remote?.includes('github')) return;
            const url = parseGitUrl(scmTraceabilityData.remote);
            return `https://${url.host}/${url.owner}/${url.name}/commit/${scmTraceabilityData.commit}`;
        };

        const commitUrl = buildUrlForGithub();

        const commitHash = scmTraceabilityData.commit.substring(0, 8);

        return (
            <div className="dot-card-spacing">
                <DotTypography variant="body1">
                    <strong>Git version: </strong>
                    {commitUrl ? (
                        <DotLink href={commitUrl} target="_blank">
                            {commitHash}
                        </DotLink>
                    ) : (
                        <>{commitHash}</>
                    )}
                </DotTypography>
            </div>
        );
    };

    return (
        <DotCard className="workflow-card">
            <div>
                <DotCardHeader avatar={cardLogo} subheader={author ? `by ${author}` : undefined} subheaderSize="large" title={title} titleSize="large" />
                <DotCardContent>
                    <div className="dot-card-spacing">
                        <DotTypography variant="body1">{description}</DotTypography>
                    </div>
                    {renderCommitLink()}
                    <div className="dot-card-spacing">
                        <DotPill backgroundcolor="white" icon={<DotIcon iconId="folder" />} label={folderTitle} variant="outlined" />
                    </div>
                </DotCardContent>
            </div>
            <DotCardFooter>
                <DotButton onClick={() => setDialogOpen(true)} type="outlined">
                    Run workflow
                </DotButton>
            </DotCardFooter>
            {isDialogOpen && (
                <DotDialog
                    cancelButtonProps={{ label: 'Cancel' }}
                    className="workflow-card-folder-dialog"
                    closeIconVisible={true}
                    closeOnClickAway={true}
                    onCancel={handleOnCancel}
                    onSubmit={handleOnRun}
                    open={isDialogOpen}
                    submitButtonProps={{ label: 'Run workflow', disabled: isNil(selectedFolderId) }}
                    title="Choose folder"
                >
                    <DotTypography className="persistent-label">
                        Select the folder where workflow <strong>{title}</strong> will be run.
                    </DotTypography>
                    <XlReactWidgetTreeSelect data={folders} onModelChange={handleSelectFolder} selectedFolderId={selectedFolderId} />
                </DotDialog>
            )}
        </DotCard>
    );
};
