import React from 'react';
import { DotAlertBanner, DotButton, DotCardFooter, DotDialog, DotIconButton, DotInputText, DotTablePagination } from '@digital-ai/dot-components';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { XlReactWidgetTreeSelect } from 'xl-react-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { WorkflowCatalog } from './workflow-catalog.component';
import { Workflow } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import { WorkflowCard } from './workflow-card.component';
import { WorkflowPagination, workflowTile, WorkflowTileState } from '../../ducks/workflow-tile.reducer';
import { WorkflowSkeletons } from './skeletons/workflow-skeletons.component';
import { WorkflowCategoriesSkeleton } from './skeletons/workflow-categories-skeleton.component';

const { runWorkflow, filterWorkflowTile } = workflowTile.actions;

describe('WorkflowCatalog', () => {
    const initialPagination: WorkflowPagination = {
        itemsPerPage: 10,
        page: 0,
    };
    const cloudWorkflow: Workflow = {
        id: 'Release1',
        title: 'My cloud workflow',
        description: 'i do stuff in the sky',
        categories: ['Cloud'],
        author: 'John Would',
        scmTraceabilityData: {},
    } as never;

    const deployWorkflow: Workflow = {
        id: 'Release2',
        title: 'My deploy workflow',
        description: 'i do stuff in the sass',
        categories: ['Deployment'],
        author: 'John Wouldnt',
        scmTraceabilityData: {},
    } as never;

    const mysteryWorkflow: Workflow = {
        id: 'Release3',
        title: 'My mystery workflow',
        description: 'i do nothing',
        categories: [],
        scmTraceabilityData: {},
    } as never;

    const dispatch = jest.fn();
    const initialState = {
        workflowTile: {
            categories: [
                { id: '1', title: 'Deployment', active: true },
                { id: '2', title: 'Cloud', active: true },
            ],
            workflows: [cloudWorkflow, deployWorkflow, mysteryWorkflow],
            folders: [],
            isLoadingWorkflows: false,
            isLoadingCategories: false,
            isLoadingFolders: false,
            totalWorkflows: 3,
            workflowTileSearch: { ...initialPagination },
        } as WorkflowTileState,
    };

    let wrapper: ReactWrapper;

    const mount = (state = initialState) => {
        wrapper = mountWithStoreAndTheme(<WorkflowCatalog />, dispatch, state);
    };

    const clickOnNextPage = () => wrapper.find('button[title="Go to next page"]').simulate('click');

    const clickOnCategory = (category: string, check: boolean) =>
        wrapper.find(`input[aria-label="${category}"]`).simulate('change', { target: { checked: check } });
    const filterByAuthor = (author: string) =>
        wrapper
            .findWhere((node) => node.is(DotInputText) && node.props().id === 'authored-by')
            .props()
            .onChange({ target: { value: author } });

    const filterByText = (text: string) =>
        wrapper
            .findWhere((node) => node.is(DotInputText) && node.props().id === 'search')
            .props()
            .onChange({ target: { value: text } });

    const getWorkflowCards = () => wrapper.find(WorkflowCard);
    const getDotTablePagination = () => wrapper.find(DotTablePagination);

    const openDialog = (index: number) => wrapper.find(DotCardFooter).at(index).find(DotButton).invoke('onClick')?.('' as never);
    const getDialog = () => wrapper.find(DotDialog);
    const getAlert = () => getDialog().find(DotAlertBanner);
    const openAlertDetails = () => getAlert().find(DotIconButton).invoke('onClick')?.('' as never);
    const getFolderDropdown = () => wrapper.find(XlReactWidgetTreeSelect);
    const getWorkflowSkeletons = () => wrapper.find(WorkflowSkeletons);
    const getCategorySkeletons = () => wrapper.find(WorkflowCategoriesSkeleton);

    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const selectFolder = (selectedId: string | undefined) => getFolderDropdown().invoke('onModelChange')?.({ selectedId });

    const getSubmitDialogButton = () => getDialog().find(DotButton).at(1);
    const submitDialog = () => getDialog().invoke('onSubmit')?.('' as never);
    const cancelDialog = () => getDialog().invoke('onCancel')?.('' as never);

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should dispatch init when mounted', () => {
        mount();
        expect(dispatch).toHaveBeenCalledWith({ payload: undefined, type: 'workflowTile/init' });
    });

    it('should execute correct dispatch action when filtering by categories', () => {
        mount();
        expect(getWorkflowCards()).toHaveLength(3);
        clickOnCategory('Cloud', true);
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ categories: ['Cloud'], ...initialPagination }));
        clickOnCategory('Deployment', true);
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ categories: ['Cloud', 'Deployment'], ...initialPagination }));
    });

    it('should execute correct dispatch action when changing page', () => {
        mount({
            ...initialState,
            workflowTile: {
                ...initialState.workflowTile,
                workflows: [
                    cloudWorkflow,
                    deployWorkflow,
                    mysteryWorkflow,
                    cloudWorkflow,
                    deployWorkflow,
                    mysteryWorkflow,
                    cloudWorkflow,
                    deployWorkflow,
                    mysteryWorkflow,
                    cloudWorkflow,
                    deployWorkflow,
                    mysteryWorkflow,
                ],
                totalWorkflows: 11,
            },
        });
        clickOnNextPage();
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ page: initialPagination.page + 1, itemsPerPage: initialPagination.itemsPerPage }));
    });

    it('should display correct numbers in items per page dropdown', () => {
        mount();
        const tablePaginationProps = getDotTablePagination().props();
        expect(tablePaginationProps.rowsPerPageOptions).toStrictEqual([10, 50, 100]);
    });

    it('should execute correct dispatch action when changing items per page', async () => {
        const newItemsPerPage = 50;
        mount();
        wrapper.find(DotTablePagination).invoke('onRowsPerPageChange')?.({ target: { value: newItemsPerPage } } as never);
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ page: initialPagination.page, itemsPerPage: newItemsPerPage }));
    });

    it('should render and filter by author', () => {
        mount();
        expect(getWorkflowCards()).toHaveLength(3);
        filterByAuthor('wouldnt');
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ author: 'wouldnt', ...initialPagination }));
    });

    it('should render and filter by text', () => {
        mount();
        expect(getWorkflowCards()).toHaveLength(3);
        filterByText('i do nothing');
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ searchInput: 'i do nothing', ...initialPagination }));
        filterByText('');
        expect(dispatch).toHaveBeenLastCalledWith(filterWorkflowTile({ searchInput: '', ...initialPagination }));
    });

    it('should render correct number of workflows if category filter has empty array', () => {
        const customState = {
            ...initialState,
            workflowTile: {
                ...initialState.workflowTile,
                workflowTileSearch: {
                    categories: [],
                    ...initialPagination,
                },
            },
        };
        mount(customState);
        expect(getWorkflowCards()).toHaveLength(3);
    });

    it('should handle run button', () => {
        mount();
        dispatch.mockReset();
        openDialog(0);
        expect(dispatch).toHaveBeenCalledWith({
            payload: 'Release2',
            type: 'workflowTile/setDialogOpen',
        });
    });

    it('should render skeletons when categories are loading', () => {
        const customState = {
            ...initialState,
            workflowTile: {
                ...initialState.workflowTile,
                isLoadingCategories: true,
            },
        };
        mount(customState);

        expect(getCategorySkeletons()).toExist();
        expect(getWorkflowSkeletons()).toExist();
        expect(getWorkflowCards()).not.toExist();
    });

    it('should render skeletons when workflows are loading', () => {
        const customState = {
            ...initialState,
            workflowTile: {
                ...initialState.workflowTile,
                isLoadingWorkflows: true,
            },
        };
        mount(customState);
        expect(getWorkflowSkeletons()).toExist();
        expect(getWorkflowCards()).not.toExist();
    });

    describe('run workflow dialog', () => {
        beforeEach(() => {
            dispatch.mockReset();
            mount({ ...initialState, workflowTile: { ...initialState.workflowTile, workflowDialogOpen: 'Release1' } });
        });

        it('should render the messages', () => {
            expect(getDialog()).toIncludeText('Choose folder');
            expect(getDialog()).toIncludeText('Folder name');
            expect(getDialog()).toIncludeText('Select the folder where workflow My cloud workflow will be run');
        });

        it('should render the folders', () => {
            expect(getDialog()).toExist();
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(getFolderDropdown().props().data).toBe(initialState.workflowTile.folders);
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            expect(getFolderDropdown().props().selectedFolderId).toBe(cloudWorkflow.folderId);
        });

        it('should hide the dialog if cancelled', () => {
            expect(getDialog()).toExist();
            cancelDialog();
            expect(dispatch).toHaveBeenCalledWith({ payload: undefined, type: 'workflowTile/setDialogOpen' });
        });

        it('should call the run workflow action', () => {
            selectFolder('Applications/Folder1');
            submitDialog();
            expect(dispatch).toHaveBeenCalledWith(
                runWorkflow({
                    workflow: cloudWorkflow,
                    folderId: 'Applications/Folder1',
                }),
            );
        });

        it('should not run the workflow is folder is not selected', () => {
            selectFolder(undefined);
            expect(getDialog().props().submitButtonProps?.disabled).toBe(true);
            dispatch.mockReset();
            submitDialog();
            expect(dispatch).not.toHaveBeenCalled();
        });

        describe('error handling', () => {
            it('should not render the error message if there is no error', () => {
                mount();
                expect(getAlert()).not.toExist();
            });

            it('should render the error message if there is error', () => {
                mount({
                    workflowTile: {
                        ...initialState.workflowTile,
                        workflowDialogOpen: 'Release1',
                        workflowDialogError: 'whops',
                    },
                });
                expect(getAlert()).toExist();
                expect(getAlert()).toIncludeText('You do not have enough permissions to run a workflow in the selected folder.');
                expect(getAlert()).not.toIncludeText('whops');
                openAlertDetails();
                expect(getAlert()).toIncludeText('You do not have enough permissions to run a workflow in the selected folder.');
                expect(getAlert()).toIncludeText('whops');
            });

            it('should clean the error when selecting folder', () => {
                mount({
                    workflowTile: {
                        ...initialState.workflowTile,
                        workflowDialogOpen: 'Release1',
                        workflowDialogError: 'whops',
                    },
                });
                selectFolder('Applications/Folder1');
                expect(dispatch).toHaveBeenCalledWith({ payload: undefined, type: 'workflowTile/setDialogError' });
            });

            it('should disable run button if there is error', () => {
                mount({
                    workflowTile: {
                        ...initialState.workflowTile,
                        workflowDialogOpen: 'Release1',
                        workflowDialogError: 'whops',
                    },
                });
                expect(getAlert()).toExist();
                expect(getSubmitDialogButton()).toBeDisabled();
            });
        });
    });
});
