import { initialState, workflowTile, WorkflowTileSearch } from './workflow-tile.reducer';

const {
    filterWorkflowTile,
    init,
    runWorkflow,
    setCategories,
    setDialogError,
    setDialogOpen,
    setFolders,
    setTotalWorkflows,
    setWorkflowTileSearch,
    setWorkflows,
} = workflowTile.actions;
describe('workflowTile.reducer', () => {
    it('should have initial state', () => {
        expect(workflowTile.name).toBe('workflowTile');
        expect(Object.keys(workflowTile.actions)).toStrictEqual([
            'init',
            'setCategories',
            'setFolders',
            'setWorkflows',
            'filterWorkflowTile',
            'setWorkflowTileSearch',
            'setDialogOpen',
            'setDialogError',
            'setTotalWorkflows',
            'runWorkflow',
        ]);
        expect(workflowTile.reducer(undefined, { type: 'default' })).toMatchObject(initialState);
    });

    describe('reducers', () => {
        const afterInitState = workflowTile.reducer(initialState, init());

        describe('init', () => {
            it('should update state', () => {
                const state = workflowTile.reducer(initialState, init());
                expect(state).toStrictEqual(afterInitState);
            });
        });

        describe('setCategories', () => {
            it('should update state', () => {
                const state = workflowTile.reducer(afterInitState, setCategories(['a', 'b'] as never));
                expect(state).toStrictEqual({
                    categories: ['a', 'b'],
                    folders: [],
                    isLoadingCategories: false,
                    isLoadingFolders: true,
                    isLoadingWorkflows: true,
                    totalWorkflows: 0,
                    workflowDialogError: undefined,
                    workflowDialogOpen: undefined,
                    workflows: [],
                });
            });
        });
        describe('setFolders', () => {
            it('should update state', () => {
                const state = workflowTile.reducer(afterInitState, setFolders(['a', 'b'] as never));
                expect(state).toStrictEqual({
                    categories: [],
                    folders: ['a', 'b'],
                    isLoadingCategories: true,
                    isLoadingFolders: false,
                    isLoadingWorkflows: true,
                    totalWorkflows: 0,
                    workflowDialogError: undefined,
                    workflowDialogOpen: undefined,
                    workflows: [],
                });
            });
        });
        describe('setWorkflows', () => {
            it('should update state', () => {
                const state = workflowTile.reducer(afterInitState, setWorkflows(['a', 'b'] as never));
                expect(state).toStrictEqual({
                    categories: [],
                    folders: [],
                    isLoadingCategories: true,
                    isLoadingFolders: true,
                    isLoadingWorkflows: false,
                    totalWorkflows: 0,
                    workflowDialogError: undefined,
                    workflowDialogOpen: undefined,
                    workflows: ['a', 'b'],
                });
            });
        });

        describe('filterWorkflowTile', () => {
            it('should do nothing with the state', () => {
                const state = workflowTile.reducer(afterInitState, filterWorkflowTile({ categories: ['a'], page: 0, itemsPerPage: 50 }));
                expect(state).toMatchObject(afterInitState);
            });
        });

        describe('setWorkflowTileSearch', () => {
            it('should update state', () => {
                const workflowTileSearch: WorkflowTileSearch = { categories: ['b', 'c'], author: 'John', page: 0, itemsPerPage: 10 };
                const state = workflowTile.reducer(afterInitState, setWorkflowTileSearch(workflowTileSearch));
                expect(state).toStrictEqual({
                    ...afterInitState,
                    workflowTileSearch,
                });
            });
        });

        describe('setDialogOpen', () => {
            it('should update state', () => {
                const { workflowDialogError, workflowDialogOpen } = workflowTile.reducer(
                    {
                        ...afterInitState,
                        workflowDialogError: 'whops',
                    },
                    setDialogOpen('Release1'),
                );
                expect(workflowDialogError).toBeUndefined();
                expect(workflowDialogOpen).toBe('Release1');
            });
        });
        describe('setDialogError', () => {
            it('should update state', () => {
                const { workflowDialogError } = workflowTile.reducer(afterInitState, setDialogError('whops'));
                expect(workflowDialogError).toBe('whops');
            });
        });

        describe('runWorkflow', () => {
            it('should not update state', () => {
                const state = workflowTile.reducer(initialState, runWorkflow({} as never));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('setTotalWorkflows', () => {
            it('should update state', () => {
                const numberOfWorkflows = 45;
                const { totalWorkflows } = workflowTile.reducer(afterInitState, setTotalWorkflows(numberOfWorkflows));
                expect(totalWorkflows).toBe(numberOfWorkflows);
            });
        });
    });
});
