import { calculateCellProps, getRandomInt, groupWorkflowsByCategory, OTHER_CATEGORY } from './index';
import { Category, Workflow } from '../../../../../../../../../../../../../core/xlr-ui/app/types';

describe('workflow catalog helper text', () => {
    describe('groupWorkflowsByCategory', () => {
        const workflows: Array<Workflow> = [
            {
                description: 'blablbala',
                categories: [],
                author: 'Digital.ai',
                tags: [],
                id: 'Release0',
                title: 'I dont have category',
            },
            {
                description: 'Create and argoCD deployment',
                categories: ['Category1'],
                author: 'Digital.ai',
                tags: [],
                id: 'Release1',
                title: 'Argo CD workflow',
            },
            {
                description: 'description',
                categories: ['Category1', 'Category2'],
                author: 'analog',
                tags: [],
                id: 'Release2',
                title: 'Authentication on prem workflow',
            },
            {
                description: 'description',
                categories: ['XXXXX'],
                author: 'Digital.ai',
                tags: [],
                id: 'Release3',
                title: 'Mystery workflow',
            },
        ] as never;

        const categories: Array<Category> = [
            {
                active: true,
                id: '1',
                title: 'Category1',
            },
            {
                active: true,
                id: '2',
                title: 'Category2',
            },
        ];

        it('should return all workflows into OTHER', () => {
            expect(groupWorkflowsByCategory([], workflows)).toStrictEqual({
                Other: [
                    {
                        author: 'Digital.ai',
                        categories: [],
                        description: 'blablbala',
                        id: 'Release0',
                        tags: [],
                        title: 'I dont have category',
                    },
                    {
                        author: 'Digital.ai',
                        categories: ['Category1'],
                        description: 'Create and argoCD deployment',
                        id: 'Release1',
                        tags: [],
                        title: 'Argo CD workflow',
                    },
                    {
                        author: 'analog',
                        categories: ['Category1', 'Category2'],
                        description: 'description',
                        id: 'Release2',
                        tags: [],
                        title: 'Authentication on prem workflow',
                    },
                    {
                        author: 'Digital.ai',
                        categories: ['XXXXX'],
                        description: 'description',
                        id: 'Release3',
                        tags: [],
                        title: 'Mystery workflow',
                    },
                ],
            });
        });

        it('should return all empty categories if there are not workflows', () => {
            expect(groupWorkflowsByCategory(categories, [])).toStrictEqual({ Category1: [], Category2: [], [OTHER_CATEGORY]: [] });
        });

        describe('filters', () => {
            const allWorkflows = {
                Category1: [
                    {
                        author: 'Digital.ai',
                        categories: ['Category1'],
                        description: 'Create and argoCD deployment',
                        id: 'Release1',
                        tags: [],
                        title: 'Argo CD workflow',
                    },
                    {
                        author: 'analog',
                        categories: ['Category1', 'Category2'],
                        description: 'description',
                        id: 'Release2',
                        tags: [],
                        title: 'Authentication on prem workflow',
                    },
                ],
                Category2: [
                    {
                        author: 'analog',
                        categories: ['Category1', 'Category2'],
                        description: 'description',
                        id: 'Release2',
                        tags: [],
                        title: 'Authentication on prem workflow',
                    },
                ],
                Other: [
                    {
                        author: 'Digital.ai',
                        categories: [],
                        description: 'blablbala',
                        id: 'Release0',
                        tags: [],
                        title: 'I dont have category',
                    },
                    {
                        author: 'Digital.ai',
                        categories: ['XXXXX'],
                        description: 'description',
                        id: 'Release3',
                        tags: [],
                        title: 'Mystery workflow',
                    },
                ],
            };

            it('should return all workflows', () => {
                expect(groupWorkflowsByCategory(categories, workflows)).toStrictEqual(allWorkflows);
            });
        });
    });

    describe('calculateCellProps', () => {
        it('should calculate the sizes correctly for lg', () => {
            expect(calculateCellProps(0).lg).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(1).lg).toStrictEqual({
                span: 4,
                start: 5,
            });
            expect(calculateCellProps(2).lg).toStrictEqual({
                span: 4,
                start: 9,
            });
            expect(calculateCellProps(3).lg).toStrictEqual({
                span: 4,
                start: 1,
            });
        });
        it('should calculate the sizes correctly for md', () => {
            expect(calculateCellProps(0).md).toStrictEqual({
                span: 6,
                start: 1,
            });
            expect(calculateCellProps(1).md).toStrictEqual({
                span: 6,
                start: 7,
            });
            expect(calculateCellProps(2).md).toStrictEqual({
                span: 6,
                start: 1,
            });
            expect(calculateCellProps(3).md).toStrictEqual({
                span: 6,
                start: 7,
            });
        });

        it('should calculate the sizes correctly for 8 columns', () => {
            expect(calculateCellProps(0).sm).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(1).sm).toStrictEqual({
                span: 4,
                start: 5,
            });
            expect(calculateCellProps(2).sm).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(3).sm).toStrictEqual({
                span: 4,
                start: 5,
            });
        });

        it('should calculate the sizes correctly for 4 columns', () => {
            expect(calculateCellProps(0).xs).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(1).xs).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(2).xs).toStrictEqual({
                span: 4,
                start: 1,
            });
            expect(calculateCellProps(3).xs).toStrictEqual({
                span: 4,
                start: 1,
            });
        });
    });

    describe('getRandomInt', () => {
        it('should display number in given range', () => {
            const numbers = Array.from({ length: 100 }, () => getRandomInt(0, 5));
            numbers.every((number: number): void => {
                expect(number).toBeGreaterThanOrEqual(0);
                expect(number).toBeLessThanOrEqual(5);
            });
        });
    });
});
