import React from 'react';
import { DotAvatar, DotButton, DotCardContent, DotCardFooter, DotCardHeader, DotIcon, DotLink, DotPill } from '@digital-ai/dot-components';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { Workflow } from '../../../../../../../../../../../../core/xlr-ui/app/types';
import { WorkflowCard } from './workflow-card.component';

describe('WorkflowCard', () => {
    const onClick = jest.fn();

    let wrapper: ReactWrapper;

    const defaultWorkflow: Workflow = {
        title: 'Create ArgoCD',
        author: 'me',
        folderId: 'Applications/FolderSamples',
        logo: {
            id: 'logoId',
        },
        id: 'Applications/FolderSamples/Release1',
        categories: [''],
        folderTitle: 'Samples and tutorials',

        scmTraceabilityData: {
            commit: 'dae95ce4de378be321c299ad7d71a28f4c429f61',
            remote: 'https://github.com/angel-git/release-devops.git',
        },
        description: 'my description',
        tags: [''],
    };

    const mount = (workflow: Workflow = defaultWorkflow) => {
        wrapper = mountWithTheme(<WorkflowCard onClick={onClick} workflow={workflow} />);
    };

    const getCardHeader = () => wrapper.find(DotCardHeader);
    const getCardContent = () => wrapper.find(DotCardContent);
    const getFolderPill = () => wrapper.find(DotPill);

    const clickOnRun = () => wrapper.find(DotCardFooter).find(DotButton).invoke('onClick')?.('' as never);

    describe('render', () => {
        it('should render the logo if present', () => {
            mount();
            expect(getCardHeader().find(DotAvatar).props().imageSrc).toBe('api/v1/templates/logo/logoId');
        });

        it('should render empty avatar if not present', () => {
            mount({ ...defaultWorkflow, logo: null });
            expect(getCardHeader().find(DotAvatar).props().text).toBe('');
        });

        it('should render the author if present', () => {
            mount();
            expect(getCardHeader().props().subheader).toBe('by me');
        });

        it('should not render the author if not present', () => {
            mount({ ...defaultWorkflow, author: null });
            expect(getCardHeader().props().subheader).toBeUndefined();
        });

        it('should render the title', () => {
            mount();
            expect(getCardHeader().props().title).toBe('Create ArgoCD');
        });

        it('should render the description', () => {
            mount();
            expect(getCardContent()).toIncludeText('my description');
        });

        it('should not render the commit if not present', () => {
            mount({ ...defaultWorkflow, scmTraceabilityData: { commit: null, remote: null } });
            expect(getCardContent()).not.toIncludeText('Git version:');
            expect(getCardContent().find(DotLink)).not.toExist();
        });

        it('should render the commit if present', () => {
            mount();
            expect(getCardContent()).toIncludeText('Git version:');
            expect(getCardContent()).toIncludeText('dae95ce4');
            expect(getCardContent().find(DotLink).props().href).toBe(
                'https://github.com/angel-git/release-devops/commit/dae95ce4de378be321c299ad7d71a28f4c429f61',
            );
        });

        it('should not render the commit link if is not from github', () => {
            mount({
                ...defaultWorkflow,
                scmTraceabilityData: {
                    commit: 'dae95ce4de378be321c299ad7d71a28f4c429f61',
                    remote: 'https://gitlab.com/angel-git/release-devops.git',
                },
            });
            expect(getCardContent()).toIncludeText('Git version:');
            expect(getCardContent()).toIncludeText('dae95ce4');
            expect(getCardContent().find(DotLink)).not.toExist();
        });

        it('should render the folder', () => {
            mount();
            const folderProps = getFolderPill().props();
            const folderIcon = getFolderPill().find(DotIcon);
            expect(folderProps.backgroundcolor).toBe('white');
            expect(folderProps.variant).toBe('outlined');
            expect(folderProps.label).toBe('Samples and tutorials');
            expect(folderIcon.props().iconId).toBe('folder');
        });
    });

    it('should handle run workflow button', () => {
        mount();
        clickOnRun();
        expect(onClick).toHaveBeenCalledWith('Applications/FolderSamples/Release1');
    });
});
