package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.applications.management.repository.persistence.ManagedApplicationSchema.MANAGED_APPLICATION
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.HomeTilesSqlBuilder
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.HomeTilesSqlBuilder.{lastEditedByUserAlias, releasesCounterAlias}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport, Utils}
import com.xebialabs.xlrelease.security.XLReleasePermissions
import com.xebialabs.xlrelease.security.XLReleasePermissions.AUDIT_ALL
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Service

import java.util.Date
import scala.jdk.CollectionConverters._

case class TemplateHomeTileRow(releaseId: String,
                               releaseTitle: String,
                               folderName: String,
                               folderId: String,
                               folderPath: String,
                               lastEditedByUser: Date,
                               releaseCount: Int)

case class ReleaseHomeTileRow(releaseId: String,
                              releaseTitle: String,
                              releaseStatus: String,
                              startDate: Date,
                              endDate: Date,
                              folderName: String,
                              folderId: String,
                              folderPath: String,
                              lastEditedByUser: Date)

case class AppHomeTileRow(appId: String,
                          appName: String,
                          appEnvironment: String,
                          appCreateDate: Date,
                          folderName: String,
                          folderId: String,
                          folderPath: String)

case class AppHomeTileFolderCount(amount: Int,
                          folderName: String,
                          folderId: String,
                          folderPath: String)


@Service
class HomeTilesQueryService(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                            @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                            folderPersistence: FolderPersistence,
                            permissionEnforcer: PermissionEnforcer,
                            roleService: RoleService) extends Utils with PersistenceSupport {

  private val sqlBuilder = new HomeTilesSqlBuilder()

  def findLastEditedTemplates(username: String): Seq[TemplateHomeTileRow] = {

    val templateRowMapper: RowMapper[TemplateHomeTileRow] = (rs, _) => TemplateHomeTileRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getString(RELEASES.RELEASE_TITLE),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH),
      rs.getTimestamp(lastEditedByUserAlias),
      rs.getInt(releasesCounterAlias)
    )
    val templateQuery = sqlBuilder.lastTemplateEditedQuery(5)
    findMany(sqlQuery(templateQuery, params("username" -> username), templateRowMapper))
  }

  def findLastEditedReleases(username: String): Seq[ReleaseHomeTileRow] = {

    val releaseRowMapper: RowMapper[ReleaseHomeTileRow] = (rs, _) => ReleaseHomeTileRow(
      rs.getString(RELEASES.RELEASE_ID),
      rs.getString(RELEASES.RELEASE_TITLE),
      rs.getString(RELEASES.STATUS),
      rs.getTimestamp(RELEASES.START_DATE),
      rs.getTimestamp(RELEASES.END_DATE),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH),
      rs.getTimestamp(lastEditedByUserAlias)
    )

    val releaseQuery = sqlBuilder.lastReleaseEditedQuery(5)
    findMany(sqlQuery(releaseQuery, params("username" -> username), releaseRowMapper))
  }

  def findLastCreatedApplications(): (Seq[AppHomeTileRow], Seq[AppHomeTileFolderCount]) = {
    val appRowMapper: RowMapper[AppHomeTileRow] = (rs, _) => AppHomeTileRow(
      rs.getString(MANAGED_APPLICATION.ID),
      rs.getString(MANAGED_APPLICATION.APPLICATION_NAME),
      rs.getString(MANAGED_APPLICATION.ENVIRONMENT_TAG),
      rs.getTimestamp(MANAGED_APPLICATION.DATE_CREATED),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH)
    )

    val appCountRowMapper: RowMapper[AppHomeTileFolderCount] = (rs, _) => AppHomeTileFolderCount(
      rs.getInt(1),
      rs.getString(FOLDERS.NAME),
      rs.getString(FOLDERS.FOLDER_ID),
      rs.getString(FOLDERS.FOLDER_PATH)
    )

    // copied from SqlFolderRepository.findViewableNodesById
    val folders = if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
      folderPersistence.findById(Folder.ROOT_FOLDER_ID)
    } else {
      folderPersistence.findByIdHavingPermission(Folder.ROOT_FOLDER_ID,
        XLReleasePermissions.VIEW_FOLDER,
        authenticationToPrincipals(getAuthentication).asScala,
        getUserRoles)
    }

    val allFlatFolders = folders.topDown.map(_.uid)
    if (allFlatFolders.size == 1) {
      // only root folder, no applications for you
      (Seq.empty, Seq.empty)
    } else {
      val appHomeTileRows = findMany(sqlQuery(sqlBuilder.lastCreatedApplications(5, allFlatFolders), params(), appRowMapper))
      val appHomeTileFolderCounts = findMany(sqlQuery(sqlBuilder.countApplicationsByFolder(allFlatFolders), params(), appCountRowMapper))
      (appHomeTileRows, appHomeTileFolderCounts)
    }
  }

  def countApplications(): Int = {
    val query = sqlBuilder.countApplications()
    sqlQuery(
      query,
      params(),
      _.getInt(1)
    ).head
  }

  private def getUserRoles = {
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
  }

}
