import React, { ReactNode, useEffect, useState } from 'react';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { XlReactWidgetTreeSelect } from 'xl-react-components';
import isNil from 'lodash/isNil';
import moment from 'moment';
import { DotDialog, DotIcon, DotPill, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { ButtonWithLink } from '../../CommonCardTile/common-card-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import colors from '../../../../../../../../../../core/xlr-ui/app/js/colors';
import { WidgetFolder } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/catalog/workflow-catalog.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { applicationTile, getApplicationTileState } from './ducks/application-tile.reducer';
import { addCounterToFolders, mapFolders } from './helpers/helpers';
import { getProfile } from '../../../../../../../../../../core/xlr-ui/app/features/profile/ducks/profile.selectors';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { RegionalSettings } from '../../../../../../../../../../core/xlr-ui/app/js/locale/regional-settings';
import './active-application-tile.component.less';

const { loadAllFolders, setApplicationDialogOpen } = applicationTile.actions;

interface HomeApplication {
    appCreateDate: number;
    appEnvironment: string;
    appId: string;
    appName: string;
    folderId: string;
    folderName: string;
    folderPath: string;
}

export interface HomeApplicationFolder {
    folderId: string;
    folderName: string;
    folderPath: string;
    total: number;
}

export interface ApplicationTileData {
    applications: Array<HomeApplication>;
    folders: Array<HomeApplicationFolder>;
    total: number;
}

export interface ActiveApplicationTileProps {
    activeTileData: ApplicationTileData;
    headerEndContent?: ReactNode;
    primaryButton: ButtonWithLink;
    secondaryButton?: ButtonWithLink;
    title: string;
}

export const ActiveApplicationTile = ({ activeTileData, headerEndContent, primaryButton, secondaryButton, title }: ActiveApplicationTileProps) => {
    const dispatch = useAppDispatch();

    const { allFolders, applicationDialogOpen } = useAppSelector(getApplicationTileState);
    const { dateFormat: profileDateFormat } = useAppSelector(getProfile);
    const dateFormat = calculateDotDateFormat(RegionalSettings.getDateFormats(profileDateFormat).shortDate);
    const mappedFolders = mapFolders(activeTileData.folders);
    const [selectedFolderId, setSelectedFolderId] = useState<string | undefined>(undefined);

    useEffect(() => {
        dispatch(loadAllFolders());
    }, []);

    const handleOnViewApplications = () => {
        if (!selectedFolderId) return;
        dispatch(setApplicationDialogOpen(false));
        window.location.href = `#/folders/${selectedFolderId}/application-pipelines/applications-management`;
    };

    const handleSelectFolder = (folder: WidgetFolder) => {
        setSelectedFolderId(folder.selectedId);
    };

    const tableColumns = [
        { id: 'applications', label: 'Application', truncate: true },
        { id: 'environment', label: 'Environment', width: '136px' },
        { id: 'createdDate', label: 'Created date', width: '136px' },
    ];

    const applicationsData = activeTileData.applications.map((application) => {
        return {
            id: application.appName,
            rowData: {
                applications: (
                    <>
                        <DotTypography data-testid="row-application-title" variant="body1">
                            {application.appName}
                        </DotTypography>
                        <DotTypography className="application-folder-name" variant="body2">
                            {application.folderName}
                        </DotTypography>
                    </>
                ),
                environment: (
                    <DotPill
                        backgroundcolor={colors.successBackground}
                        bordercolor={colors.green}
                        icon={<DotIcon className="environment-infrastructure-icon" iconId="infrastructure" />}
                        label={application.appEnvironment}
                        size="small"
                        variant="outlined"
                    />
                ),
                createdDate: (
                    <DotTooltip data-testid="created-date-tooltip" placement="bottom-start" title={moment(application.appCreateDate).format(dateFormat)}>
                        <DotTypography variant="body1">{moment(application.appCreateDate).fromNow()}</DotTypography>
                    </DotTooltip>
                ),
            },
        };
    });

    const onRowClick = (id: string) => {
        // TODO open application edit page
    };

    const renderDialog = () => {
        return (
            <DotDialog
                cancelButtonProps={{ label: 'Cancel' }}
                className="card-folder-dialog"
                closeIconVisible={true}
                closeOnClickAway={true}
                closeOnSubmit={false}
                onCancel={() => dispatch(setApplicationDialogOpen(false))}
                onSubmit={() => handleOnViewApplications()}
                open={applicationDialogOpen}
                submitButtonProps={{ label: 'View applications', disabled: isNil(selectedFolderId) }}
                title="Choose folder"
            >
                <DotTypography data-testid="application-dialog-description">Select a folder to show Applications in this folder.</DotTypography>
                <DotTypography className="persistent-label" variant="subtitle2">
                    Folder name
                </DotTypography>
                <div className="xl-react-widget-tree-select-wrapper">
                    <XlReactWidgetTreeSelect
                        clearable={false}
                        data={addCounterToFolders(allFolders, mappedFolders)}
                        onModelChange={handleSelectFolder}
                        selectedFolderId={selectedFolderId}
                    />
                </div>
            </DotDialog>
        );
    };

    const mainContent = (
        <DotTable
            className="active-application-table"
            columns={tableColumns}
            data={applicationsData}
            onRowClick={(_event, id) => onRowClick(id)}
            sortable={false}
        />
    );

    return (
        <div className="active-application-tile">
            <CommonActiveTile
                avatarIcon="applications"
                content={mainContent}
                headerEndContent={headerEndContent}
                primaryButton={primaryButton}
                secondaryButton={secondaryButton}
                subtitle="Last added - Most recent 5"
                title={title}
            />
            {renderDialog()}
        </div>
    );
};
