package com.xebialabs.xlrelease.plugins.dashboard.views

import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.databind.ObjectMapper
import com.google.common.base.Preconditions.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.documentation.PublicApi
import com.xebialabs.xlrelease.domain.Task.CATEGORY_INPUT
import com.xebialabs.xlrelease.plugins.dashboard.builder.DashboardBuilder.newTile
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.variable.VariableViewHelper.{fromView, toView}

import java.util.{Map => JMap}
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

//noinspection ScalaStyle
object TileView {
  def apply(tile: Tile): TileView = new TileView(tile)

  def toTile(tileView: TileView)(implicit configurationRepository: ConfigurationRepository, objectMapper: ObjectMapper): Tile = {
    val tileDescriptor = Type.valueOf(tileView.`type`).getDescriptor
    val (propertiesWithValue, propertiesWithVariable) = tileView.getProperties.asScala
      .map {
        case (name, viewValue) =>
          val pd = tileDescriptor.getPropertyDescriptor(name)
          checkArgument(pd != null, s"Property [$name] is not part of synthetic type [${tileView.`type`}]".asInstanceOf[Object])
          name -> fromView(pd, viewValue)
      }
      .partition { case (_, (variable, _)) => variable == null }
    val properties: Map[String, Any] = propertiesWithValue.map { case (propertyName, (_, propertyValue)) =>
      convertValue(tileDescriptor, propertyName, propertyValue)
    }.toMap
    val variableMapping: Map[String, String] = propertiesWithVariable.view.mapValues(_._1).toMap
    newTile
      .withType(tileView.`type`)
      .withId(tileView.getId)
      .withTitle(tileView.getTitle)
      .withDescription(tileView.getDescription)
      .withRow(tileView.getRow)
      .withCol(tileView.getCol)
      .withWidth(tileView.getSizeX)
      .withMinSizeY(tileView.getMinSizeY)
      .withHeight(tileView.getSizeY)
      .withProperties(properties)
      .withVariableMappings(variableMapping)
      .build
  }
}

@PublicApi // Just to mark that this structure is documented and should not be changed
@JsonIgnoreProperties(ignoreUnknown = true)
class TileView(val tile: Tile) {

  @BeanProperty
  var id: String = tile.getId

  @BeanProperty
  var `type`: String = tile.getType.toString

  @BeanProperty
  var uri: String = tile.getUri

  @BeanProperty
  var detailsUri: String = tile.getDetailsUri

  @BeanProperty
  var configurationUri: String = tile.getConfigurationUri

  @BeanProperty
  var title: String = tile.getTitle

  @BeanProperty
  var description: String = tile.getDescription

  @BeanProperty
  var row: Integer = tile.getRow
  @BeanProperty
  var col: Integer = tile.getCol
  @BeanProperty
  var sizeX: Integer = tile.getWidth
  @BeanProperty
  var sizeY: Integer = tile.getHeight
  @BeanProperty
  var minSizeY: Integer = tile.getMinSizeY
  @BeanProperty
  var embedded: Boolean = tile.isEmbedded

  @BeanProperty
  var properties: JMap[String, AnyRef] = {
    tile.getType.getDescriptor.getPropertyDescriptors.asScala
      .filter(_.getCategory == CATEGORY_INPUT)
      .map(p => (p.getName, toView(p, tile.getVariableMapping.get(p.getName), p.get(tile))))
      .toMap.asJava
  }

  def this() = {
    // Tile is virtual, but Jackson needs to create an empty view object
    this(Tile.newDummy())
  }
}
