import React from 'react';
import moment from 'moment';
import { DotButton } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { CommonCardTileComponent } from '../../CommonCardTile/common-card-tile.component';
import { ActiveTileSkeleton } from '../../CommonCardTile/active-tile.skeleton';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { ApplicationTileComponent, ApplicationTileProps } from './application-tile.component';
import { applicationTile, initialState as applicationTileInitialState } from './ducks/application-tile.reducer';
import { ActiveApplicationTile } from './active-application-tile.component';
import { DATE_FORMAT_DAY_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import { workflow } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { DEFAULT_APPLICATION_SEARCH_CATEGORY } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/constants';

const { setApplicationDialogOpen } = applicationTile.actions;
const { init } = workflow.actions;

describe('ApplicationTileComponent', () => {
    const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;

    const dispatch = jest.fn();

    const title = 'Applications';
    const description = 'Test application description';

    const defaultProps: ApplicationTileProps = {
        title,
        description,
    };

    const defaultState = {
        applicationTile: { ...applicationTileInitialState, isLoading: false },
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
        workflow: { categories: [], workflows: [] },
    };

    const mount = (props: ApplicationTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ApplicationTileComponent {...props} />, dispatch, state);
    };

    const getCommonCardTileComponent = (wrapper: ReactWrapper) => wrapper.find(CommonCardTileComponent);
    const getFirstButton = (wrapper: ReactWrapper) => wrapper.find(DotButton).at(0);
    const getSecondButton = (wrapper: ReactWrapper) => wrapper.find(DotButton).at(1);
    const getActiveApplicationTile = (wrapper: ReactWrapper) => wrapper.find(ActiveApplicationTile);
    const getActiveTileSkeleton = (wrapper: ReactWrapper) => wrapper.find(ActiveTileSkeleton);

    beforeEach(() => {
        getAngularServiceSpy.mockReturnValue({
            interpolateInText: jest.fn().mockReturnValue(''),
        });

        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    it('should render with correct props', () => {
        const wrapper = mount();
        const props = getCommonCardTileComponent(wrapper).props();
        expect(props.title).toBe(title);
        expect(props.description).toBe(description);
        expect(props.imageLink).toBe('static/23.3.14/styles/img/application_empty_tile_sample.svg');
        expect(props.primaryButton.buttonCaption).toBe('Add application');
        expect(props.primaryButton.buttonType).toBe('outlined');
        expect(props.secondaryButton).toBe(undefined);
        expect(getActiveApplicationTile(wrapper)).not.toExist();
        expect(getActiveTileSkeleton(wrapper)).not.toExist();
    });

    it('should handle click event for Add application button', () => {
        const wrapper = mount();
        const firstButton = getFirstButton(wrapper);
        firstButton.invoke('onClick')?.('' as never);
        expect(dispatch).toHaveBeenCalledWith(init({ categories: [DEFAULT_APPLICATION_SEARCH_CATEGORY] }));
    });

    it('should show View applications button and handle click on it', () => {
        const wrapper = mount(defaultProps, {
            ...defaultState,
            applicationTile: {
                ...applicationTileInitialState,
                applicationTileData: {
                    ...applicationTileInitialState.applicationTileData,
                    applications: [
                        {
                            appCreateDate: 1690187050870,
                            appEnvironment: 'test environment',
                            appId: 'appId',
                            appName: 'Test App',
                            folderId: 'folderId',
                            folderName: 'test',
                            folderPath: '/',
                        },
                    ],
                },
                isLoading: false,
            },
        });

        const activeTileProps = getActiveApplicationTile(wrapper).props();
        expect(activeTileProps.primaryButton.buttonType).toBe('text');
        expect(activeTileProps.secondaryButton?.buttonCaption).toBe('View applications');
        expect(activeTileProps.secondaryButton?.buttonType).toBe('outlined');

        const secondButton = getSecondButton(wrapper);
        secondButton.invoke('onClick')?.('' as never);
        expect(dispatch).toBeCalledWith(setApplicationDialogOpen(true));
    });

    it('should show active tile skeleton if loading', () => {
        const wrapper = mount(defaultProps, {
            ...defaultState,
            applicationTile: { ...applicationTileInitialState, isLoading: true },
        });
        expect(getActiveTileSkeleton(wrapper)).toExist();
    });
});
