import React from 'react';
import { DotButton, DotIconButton, DotTypography } from '@digital-ai/dot-components';
import { DashboardDetailsHeaderContent, DashboardDetailsHeaderProps } from './dashboard-details-header.component';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { dashboardsMock } from '../../mocks';
import * as hooks from '../../../../../../../../../../../core/xlr-ui/app/features/common/hooks';
import { DashboardActionContextMenu } from './dashboard-action-context-menu.component';
import { ReleaseStatus } from '../../../../../../../../../../../core/xlr-ui/app/features/release-overview/release-status-label/release-status.component';

describe('DashboardDetailsHeaderContent', () => {
    let wrapper: ReactWrapper;
    const dashboard = dashboardsMock[0];
    const onEditButtonClick = jest.fn();
    const onExportAsPdfButtonClick = jest.fn();
    const onRefreshButtonClick = jest.fn();
    const editButtonProps = {
        onClick: onEditButtonClick,
    };
    const exportAsPdfButtonProps = {
        disabled: false,
        onClick: onExportAsPdfButtonClick,
    };
    const refreshButtonProps = {
        disabled: false,
        onClick: onRefreshButtonClick,
    };
    const onDelete = jest.fn();
    const onDuplicate = jest.fn();
    const canCreateDashboard = true;
    const canDeleteDashboard = true;

    jest.spyOn(hooks, 'useCloseSidebar');

    const defaultProps: DashboardDetailsHeaderProps = {
        canCreateDashboard,
        canDeleteDashboard,
        canEditDashboard: true,
        dashboard,
        editButtonProps,
        exportAsPdfButtonProps,
        isDesignMode: false,
        isFolderDashboard: false,
        isGlobalDashboard: true,
        onDelete,
        onDuplicate,
        refreshButtonProps,
    };

    const mount = (props = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<DashboardDetailsHeaderContent {...props} />, jest.fn(), {});
    };

    const getHeaderTitleTypography = () => wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'header-title');
    const getRefreshDashboardButton = () => wrapper.findWhere((node) => node.is(DotButton) && node.props()['data-testid'] === 'refresh-dashboard-btn');
    const getExportAsPdfButton = () => wrapper.findWhere((node) => node.is(DotButton) && node.props()['data-testid'] === 'export-as-pdf-btn');
    const getEditPropertiesButton = () => wrapper.findWhere((node) => node.is(DotButton) && node.props()['data-testid'] === 'edit-properties-btn');
    const getDotIconButton = () => wrapper.find(DotIconButton);
    const getDashboardActionContextMenu = () => wrapper.find(DashboardActionContextMenu);
    const getReleaseStatus = () => wrapper.find(ReleaseStatus);

    afterEach(() => jest.resetAllMocks());
    afterAll(() => jest.restoreAllMocks());

    it('should render correct elements with default props', () => {
        mount();
        const headerTitleTypography = getHeaderTitleTypography();
        expect(headerTitleTypography).toExist();
        const headerTitleTypographyProps = headerTitleTypography.props();
        expect(headerTitleTypographyProps.variant).toBe('h1');
        expect(headerTitleTypographyProps.noWrap).toBe(true);
        expect(headerTitleTypographyProps.children).toBe(dashboard.title);

        const refreshDashboardButton = getRefreshDashboardButton();
        expect(refreshDashboardButton).toExist();
        const refreshDashboardButtonProps = refreshDashboardButton.props();
        expect(refreshDashboardButtonProps.startIcon.props.iconId).toEqual('refresh');
        expect(refreshDashboardButtonProps.type).toBe('text');
        expect(refreshDashboardButtonProps.disabled).toBe(false);

        const exportAsPdfBtn = getExportAsPdfButton();
        expect(exportAsPdfBtn).toExist();
        const exportAsPdfBtnProps = exportAsPdfBtn.props();
        expect(exportAsPdfBtnProps.startIcon.props.iconId).toEqual('download');
        expect(exportAsPdfBtnProps.type).toBe('outlined');
        expect(exportAsPdfBtnProps.disabled).toBe(false);

        const editProperties = getEditPropertiesButton();
        expect(editProperties).not.toExist();

        const dotIconButton = getDotIconButton();
        expect(dotIconButton).toExist();
        expect(dotIconButton.props().iconId).toBe('options');

        const contextMenu = getDashboardActionContextMenu();
        expect(contextMenu).toExist();
        const contextMenuProps = contextMenu.props();
        expect(contextMenuProps.dashboard).toStrictEqual(dashboard);
        expect(contextMenuProps.hasCreateDashboardPermission).toBe(canCreateDashboard);
        expect(contextMenuProps.hasDeleteDashboardPermission).toBe(canDeleteDashboard);

        const releaseStatus = getReleaseStatus();
        expect(releaseStatus).not.toExist();
    });

    it('should render correct elements in design mode', () => {
        mount({ ...defaultProps, isDesignMode: true });

        const editProperties = getEditPropertiesButton();
        expect(editProperties).toExist();
        const editPropertiesProps = editProperties.props();
        expect(editPropertiesProps.startIcon.props.iconId).toEqual('edit');
        expect(editPropertiesProps.type).toBe('outlined');

        const refreshDashboardButton = getRefreshDashboardButton();
        expect(refreshDashboardButton).not.toExist();

        const exportAsPdfBtn = getExportAsPdfButton();
        expect(exportAsPdfBtn).not.toExist();
    });

    it('should set anchorEl as null on the DashboardActionContextMenu when onLeave handler has been executed', () => {
        mount();
        const element = document.createElement('div');
        expect(getDashboardActionContextMenu().props().anchorEl).toBeUndefined();

        const dotIconButton = getDotIconButton();
        dotIconButton.invoke('onClick')?.({ currentTarget: element } as never);

        const contextMenu = getDashboardActionContextMenu();
        expect(getDashboardActionContextMenu().props().anchorEl).toStrictEqual(element);
        contextMenu.invoke('onLeave')();

        expect(getDashboardActionContextMenu().props().anchorEl).toBeUndefined();
    });

    it('should NOT render context menu when isGlobalDashboard and canCreateDashboard are set to false', () => {
        mount({ ...defaultProps, canCreateDashboard: false, isGlobalDashboard: false });
        expect(getDashboardActionContextMenu()).not.toExist();
    });

    it('should render context menu when isGlobalDashboard and canDeleteDashboard are set to true, and canCreateDashboard is set to false', () => {
        mount({ ...defaultProps, canDeleteDashboard: true, isGlobalDashboard: true, canCreateDashboard: false });
        expect(getDashboardActionContextMenu()).toExist();
    });

    it('should render context menu when isGlobalDashboard and canDeleteDashboard are set to false but canCreateDashboard is set to true', () => {
        mount({ ...defaultProps, canDeleteDashboard: false, isGlobalDashboard: false, canCreateDashboard: true });
        expect(getDashboardActionContextMenu()).toExist();
    });

    it('should NOT render edit button (in design mode) when canEditDashboard is set to false', () => {
        mount({ ...defaultProps, isDesignMode: true, canEditDashboard: false });
        expect(getEditPropertiesButton()).not.toExist();
    });

    it('should NOT render edit button (in design mode) when canEditDashboard is set to true, but both isGlobalDashboard and isFolderDashboard are set to false', () => {
        mount({ ...defaultProps, isDesignMode: true, canEditDashboard: true, isGlobalDashboard: false, isFolderDashboard: false });
        expect(getEditPropertiesButton()).not.toExist();
    });

    it('should render ReleaseStatus component when release is defined', () => {
        const release = { status: 'IN_PROGRESS' };
        mount({ ...defaultProps, release: release as never });
        const releaseStatus = getReleaseStatus();
        expect(releaseStatus).toExist();
        expect(releaseStatus.props().status).toBe(release.status);
    });

    it('should run useCloseSidebar hook when is global dashboard', () => {
        mount({ ...defaultProps, isGlobalDashboard: true });
        expect(hooks.useCloseSidebar).toHaveBeenCalled();
    });

    it('should NOT run useCloseSidebar hook when is NOT global dashboard', () => {
        mount({ ...defaultProps, isGlobalDashboard: false });
        expect(hooks.useCloseSidebar).not.toHaveBeenCalled();
    });
});
