import React from 'react';
import { GlobalDashboardListContent } from './global-dashboard-list.component';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { dashboardsMock } from '../../../mocks';
import { globalDashboard } from '../../../ducks/global-dashboard.reducer';
import { navigation } from '../../../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { DashboardList } from './dashboard-list.component';
import * as angularAccessor from '../../../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const { executeDashboardAction, loadGlobalDashboards, searchWithFilter } = globalDashboard.actions;
const { navigateTo } = navigation.actions;

describe('GlobalDashboardList', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const dashboards = dashboardsMock;
    const hasCreateDashboardPermission = true;
    const loading = false;

    const defaultState = {
        globalDashboard: {
            titleFilter: '',
            dashboards,
            hasCreateDashboardPermission,
            loading,
        },
    };

    const mount = (state = defaultState) => {
        wrapper = mountWithStoreAndTheme(<GlobalDashboardListContent filters={{ title: 'initial filter' }} />, dispatch, state);
    };

    const getDashboardList = () => wrapper.find(DashboardList);

    beforeEach(() => {
        const authenticator = {
            hasPermission: jest.fn(),
        };
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        getAngularServiceSpy.mockReturnValue(authenticator);
    });

    afterEach(() => jest.resetAllMocks());

    it('should render correct elements when at least one dashboard is defined', () => {
        mount();
        const dashboardList = getDashboardList();
        expect(dashboardList).toExist();
        const dashboardListProps = dashboardList.props();
        expect(dashboardListProps.dashboards).toStrictEqual(dashboards);
        expect(dashboardListProps.hasCreateDashboardPermission).toBe(hasCreateDashboardPermission);
        expect(dashboardListProps.isLoading).toStrictEqual(loading);

        expect(dispatch).toHaveBeenCalledWith(
            loadGlobalDashboards({
                title: 'initial filter',
            }),
        );
    });

    it('should execute correct handler when invoking onCreate prop', () => {
        mount();
        const dashboardList = getDashboardList();
        dashboardList.invoke('onCreate')();
        expect(dispatch).toHaveBeenLastCalledWith(navigateTo({ pathSuffix: 'dashboards/new' }));
    });

    it('should execute correct handler when invoking onDelete prop', () => {
        mount();
        const dashboardId = dashboards[0].id as string;
        const dashboardList = getDashboardList();
        dashboardList.invoke('onDelete')(dashboardId);
        expect(dispatch).toHaveBeenLastCalledWith(executeDashboardAction({ actionId: 'DELETE', dashboard: dashboards[0] }));
    });

    it('should execute correct handler when invoking onDuplicate prop', () => {
        mount();
        const dashboard = dashboards[0];
        const dashboardList = getDashboardList();
        dashboardList.invoke('onDuplicate')(dashboard);
        expect(dispatch).toHaveBeenLastCalledWith(executeDashboardAction({ actionId: 'DUPLICATE', dashboard }));
    });

    it('should trigger dispatch when filtering', () => {
        mount();
        const dashboardList = getDashboardList();
        dashboardList.invoke('onFilterChange')?.('DASHBOARD 1');
        expect(dispatch).toHaveBeenCalledWith(searchWithFilter('DASHBOARD 1'));
    });
});
