import React, { ReactNode, useState } from 'react';
import { DotConfirmationDialog, DotIcon } from '@digital-ai/dot-components';
import { Dashboard } from '../../../types';
import { DashboardDeleteDialogContent } from './dashboard-delete-dialog-content.component';
import { DashboardDuplicateDialogContent } from './dashboard-duplicate-dialog-content.component';
import './dashboard-dialog.component.less';

export interface DashboardDialogProps {
    dashboard: Dashboard;
    dashboardActionType: 'DELETE' | 'DUPLICATE';
    dialogTitle: string;
    onDialogClose: () => void;
    onDialogSubmit: (dashboardTitle?: string) => void;
}

export const LENGTH_ERROR_MESSAGE = 'Input exceeds limit. Maximum length is 255 characters. Please shorten your text before submitting.';
export const TITLE_MAX_LENGTH = 255;

export const DashboardDialog = ({ dashboard, dashboardActionType, dialogTitle, onDialogSubmit, onDialogClose }: DashboardDialogProps) => {
    const [internalDuplicateDashboardTitle, setInternalDuplicateDashboardTitle] = useState<string>(`${dashboard.title} (copy)`);
    const errorMessage = internalDuplicateDashboardTitle.length > TITLE_MAX_LENGTH ? LENGTH_ERROR_MESSAGE : undefined;
    const isDeleteAction = dashboardActionType === 'DELETE';

    const getSubmitButtonProps = (): { disabled?: boolean; label: string; startIcon: ReactNode; type?: 'primary' | 'destructive' } => {
        return isDeleteAction
            ? { label: 'Delete dashboard', type: 'destructive', startIcon: <DotIcon data-testid="delete-start-icon" iconId="delete" /> }
            : {
                  label: 'Duplicate dashboard',
                  disabled: !!errorMessage || internalDuplicateDashboardTitle === '',
                  type: 'primary',
                  startIcon: <DotIcon data-testid="duplicate-start-icon" iconId="duplicate" />,
              };
    };

    const getDialogMessage = () => {
        return isDeleteAction ? (
            <DashboardDeleteDialogContent dashboard={dashboard} />
        ) : (
            <DashboardDuplicateDialogContent
                dashboard={dashboard}
                errorMessage={errorMessage}
                onNameChange={handleNameChange}
                title={internalDuplicateDashboardTitle}
            />
        );
    };

    const handleNameChange = (newTitle: string) => {
        setInternalDuplicateDashboardTitle(newTitle.trim());
    };

    const handleSubmit = () => {
        isDeleteAction ? onDialogSubmit() : onDialogSubmit(internalDuplicateDashboardTitle);
    };

    return (
        <DotConfirmationDialog
            className="dashboard-confirmation-dialog"
            message={getDialogMessage()}
            onCancel={onDialogClose}
            onSubmit={handleSubmit}
            open={true}
            submitButtonProps={getSubmitButtonProps()}
            title={dialogTitle}
        />
    );
};
