import { SagaIterator } from 'redux-saga';
import { all, call, put, takeLatest } from 'redux-saga/effects';

import { globalDashboard } from './global-dashboard.reducer';
import {
    executeDashboardActionSaga,
    globalDashboardSaga,
    loadDashboards,
    loadGlobalDashboardsSaga,
    searchWithFilterSaga,
    toastr,
} from './global-dashboard.saga';
import { hasPermission } from '../../../../../../../../../../core/xlr-ui/app/js/auth/authenticator.saga';
import { httpGET } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { Dashboard } from '../types';
import getAngularService from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const { executeDashboardAction, setGlobalDashboards, loadGlobalDashboards, setCreateDashboardPermission, searchWithFilter, setGlobalFilter, setLoading } =
    globalDashboard.actions;

describe('globalDashboardSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = globalDashboardSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(loadGlobalDashboards, loadGlobalDashboardsSaga),
                takeLatest(executeDashboardAction, executeDashboardActionSaga),
                takeLatest(searchWithFilter, searchWithFilterSaga),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('loadGlobalDashboardsSaga', () => {
        const action = { type: '', payload: { title: 'filter' } };
        it('should set permission if user has dashboard#create', () => {
            const gen: SagaIterator = loadGlobalDashboardsSaga(action);
            expect(gen.next().value).toStrictEqual(put(setLoading(true)));
            expect(gen.next().value).toStrictEqual(call(hasPermission, 'dashboard#create'));
            expect(gen.next(true).value).toStrictEqual(put(setCreateDashboardPermission(true)));
        });

        it('should not update permission if user has no permissions', () => {
            const gen: SagaIterator = loadGlobalDashboardsSaga(action);
            expect(gen.next().value).toStrictEqual(put(setLoading(true)));
            expect(gen.next().value).toStrictEqual(call(hasPermission, 'dashboard#create'));
            expect(gen.next(false).value).toStrictEqual(call(httpGET, 'dashboards?title=filter'));
        });

        it('should set dashboards', () => {
            const gen: SagaIterator = loadGlobalDashboardsSaga(action);
            expect(gen.next().value).toStrictEqual(put(setLoading(true)));
            expect(gen.next().value).toStrictEqual(call(hasPermission, 'dashboard#create'));
            expect(gen.next(true).value).toStrictEqual(put(setCreateDashboardPermission(true)));
            expect(gen.next().value).toStrictEqual(call(httpGET, 'dashboards?title=filter'));
            expect(gen.next({ data: [] }).value).toStrictEqual(put(setGlobalDashboards([])));
            expect(gen.next().value).toStrictEqual(put(setGlobalFilter('filter')));
            expect(gen.next().value).toStrictEqual(put(setLoading(false)));
            expect(gen.next().done).toStrictEqual(true);
        });
    });

    describe('executeDashboardActionSaga', () => {
        const xlrDashboardService = { delete: jest.fn(), create: jest.fn() };
        const dashboard: Dashboard = {
            id: '1',
            title: 'My dashboard',
            metadata: { security: { permissions: ['dashboard#edit'], teams: [] } },
        } as never;

        const clientSettings = {
            getGlobalDashboardListFilters: () => ({ title: 'my filter' }),
        } as never;

        it('should call proper sagas when deleting a dashboard ', () => {
            const gen: SagaIterator = executeDashboardActionSaga({ payload: { actionId: 'DELETE', dashboard }, type: 'testType' });
            expect(gen.next(true).value).toStrictEqual(call(getAngularService, 'xlrelease.XlrDashboardService'));
            expect(gen.next(xlrDashboardService).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call([xlrDashboardService, xlrDashboardService.delete], dashboard));
            expect(gen.next().value).toStrictEqual(call(loadDashboards, 'my filter'));
            expect(gen.next().value).toStrictEqual(call(toastr.success, 'Dashboard successfully deleted.'));
            expect(gen.next().done).toStrictEqual(true);
        });

        it('should call proper sagas when duplicating a dashboard ', () => {
            const gen: SagaIterator = executeDashboardActionSaga({
                payload: { actionId: 'DUPLICATE', dashboard },
                type: 'testType',
            });
            expect(gen.next(true).value).toStrictEqual(call(getAngularService, 'xlrelease.XlrDashboardService'));
            expect(gen.next(xlrDashboardService).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call([xlrDashboardService, xlrDashboardService.create], dashboard));
            expect(gen.next().value).toStrictEqual(call(loadDashboards, 'my filter'));
            expect(gen.next().value).toStrictEqual(call(toastr.success, 'Dashboard successfully duplicated.'));
            expect(gen.next().done).toStrictEqual(true);
        });

        it('should handle error', () => {
            const gen: SagaIterator = executeDashboardActionSaga({
                payload: { actionId: 'DUPLICATE', dashboard },
                type: 'testError',
            });
            expect(gen.next(true).value).toStrictEqual(call(getAngularService, 'xlrelease.XlrDashboardService'));
            expect(gen.next(xlrDashboardService).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call([xlrDashboardService, xlrDashboardService.create], dashboard));
            expect(gen.throw?.({ response: { data: 'whops' } }).value).toStrictEqual(call(toastr.error, 'Failed to duplicate a dashboard. Error: whops'));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('searchWithFilterSaga', () => {
        const clientSettings = {
            setGlobalDashboardListFilters: jest.fn(),
        };

        const filtersQueryParams = {
            update: jest.fn(),
        };

        it('should call proper sagas', () => {
            const gen: SagaIterator = searchWithFilterSaga({ type: '', payload: 'my filter' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(filtersQueryParams).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(loadDashboards, 'my filter'));
            expect(gen.next().done).toBeTruthy();
            expect(clientSettings.setGlobalDashboardListFilters).toHaveBeenCalledWith({ title: 'my filter' });
            expect(filtersQueryParams.update).toHaveBeenCalledWith({ title: 'my filter' });
        });
    });
});
