import { all, call, put, takeLatest } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { AxiosError } from 'axios';

import { globalDashboard } from './global-dashboard.reducer';
import { httpGET } from '@xlr-ui/app/features/common/services/http';
import { hasPermission } from '@xlr-ui/app/js/auth/authenticator.saga';
import { Dashboard, ExecuteDashboardActionRequest } from '../types';
import { PayloadAction } from '@reduxjs/toolkit';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { XlrDashboardService } from '@xlr-ui/app/features/tasks/types/angular';
import ToastrFactory from '@xlr-ui/app/js/util/toastrFactory';
import { GlobalDashboardFilters } from '../components/dashboard/list/global-dashboard-list.component';
import { ClientSettings, FilterQueryParams } from '@xlr-ui/app/types';

const { executeDashboardAction, loadGlobalDashboards, setGlobalDashboards, setCreateDashboardPermission, setGlobalFilter, searchWithFilter, setLoading } =
    globalDashboard.actions;

export const toastr = ToastrFactory();

interface GlobalDashboardClientSettings extends ClientSettings {
    getGlobalDashboardListFilters: () => GlobalDashboardFilters;
    setGlobalDashboardListFilters: (filter: GlobalDashboardFilters) => void;
}

export function* loadGlobalDashboardsSaga(action: PayloadAction<GlobalDashboardFilters>) {
    const { title } = action.payload;
    yield put(setLoading(true));
    const hasDashboardCreatePermission: boolean = yield call(hasPermission, 'dashboard#create');
    if (hasDashboardCreatePermission) {
        yield put(setCreateDashboardPermission(true));
    }
    yield* loadDashboards(title);
    yield put(setLoading(false));
}

export function* loadDashboards(title: string | undefined) {
    const url = title ? `dashboards?title=${title}` : 'dashboards';
    const { data }: IHttpResponse<Dashboard[]> = yield call(httpGET, url);
    yield put(setGlobalDashboards(data));
    yield put(setGlobalFilter(title || ''));
}

export function* executeDashboardActionSaga(action: PayloadAction<ExecuteDashboardActionRequest>) {
    const { actionId, dashboard } = action.payload;
    const xlrDashboardService: XlrDashboardService = yield call(getAngularService, 'xlrelease.XlrDashboardService');
    const clientSettings: GlobalDashboardClientSettings = yield call(getAngularService, 'ClientSettings');
    const filter = clientSettings.getGlobalDashboardListFilters();
    try {
        if (actionId === 'DELETE') {
            yield call([xlrDashboardService, xlrDashboardService.delete], dashboard);
            yield call(loadDashboards, filter.title);
            yield call(toastr.success, 'Dashboard successfully deleted.');
        } else if (actionId === 'DUPLICATE') {
            yield call([xlrDashboardService, xlrDashboardService.create], dashboard);
            yield call(loadDashboards, filter.title);
            yield call(toastr.success, 'Dashboard successfully duplicated.');
        }
    } catch (e) {
        const err = e as AxiosError;
        const errorData = err.response?.data as string;
        yield call(toastr.error, `Failed to ${actionId === 'DELETE' ? 'delete' : 'duplicate'} a dashboard. Error: ${errorData}`);
    }
}

export function* searchWithFilterSaga(action: PayloadAction<string>) {
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    const clientSettings: GlobalDashboardClientSettings = yield call(getAngularService, 'ClientSettings');
    const title = action.payload;
    const filters: GlobalDashboardFilters = { title };
    clientSettings.setGlobalDashboardListFilters(filters);
    FiltersQueryParams.update(filters);
    yield call(loadDashboards, title);
}

export function* globalDashboardSaga() {
    yield all([
        takeLatest(loadGlobalDashboards, loadGlobalDashboardsSaga),
        takeLatest(executeDashboardAction, executeDashboardActionSaga),
        takeLatest(searchWithFilter, searchWithFilterSaga),
    ]);
}
