package com.xebialabs.xlrelease.plugins.dashboard.cache

import com.github.benmanes.caffeine.cache.{Cache, CacheLoader, Caffeine}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.config.CacheManagementConstants.TILE_CACHE_MANAGER
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import grizzled.slf4j.Logging
import org.springframework.cache.CacheManager
import org.springframework.cache.caffeine.CaffeineCacheManager
import org.springframework.cache.interceptor.{BasicOperation, CacheOperationInvocationContext, CacheResolver}
import org.springframework.context.annotation.{Bean, Configuration}

import java.util.concurrent.TimeUnit
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object TileCacheConfiguration {
  final val TILE_CACHE = "tileCache"
  final val TILE_CACHE_KEY_GENERATOR = "tileCacheKeyGenerator"
  final val TILE_CACHE_RESOLVER = "tileCacheResolver"
}

@Configuration
class TileCacheConfiguration extends Logging {

  @Bean(name = Array(TileCacheConfiguration.TILE_CACHE_KEY_GENERATOR))
  def tileCacheKeyGenerator(): TileCacheKeyGenerator = {
    new TileCacheKeyGenerator()
  }

  @Bean(name = Array(TileCacheConfiguration.TILE_CACHE_RESOLVER))
  def tileCacheResolver(): CacheResolver = {
    context: CacheOperationInvocationContext[_ <: BasicOperation] => {
      val tileTypeName = context.getArgs.find(_.isInstanceOf[Tile]) match {
        case Some(tile) => tile.asInstanceOf[Tile].getType.toString
        case None => TileCacheConfiguration.TILE_CACHE
      }
      List(tileCacheManager().getCache(tileTypeName)).asJava
    }
  }

  @Bean(name = Array(TILE_CACHE_MANAGER))
  def tileCacheManager(): CacheManager = {
    val cacheManager: CaffeineCacheManager = new CaffeineCacheManager() {
      override def createNativeCaffeineCache(name: String): Cache[AnyRef, AnyRef] = {
        Try(Type.valueOf(name)) match {
          case Success(matchedTileType) =>
            // create a cache for this specific type
            cacheBuilderFrom(matchedTileType).build[AnyRef, AnyRef]()
          case Failure(_) =>
            logger.warn(s"Unable to find tile type with a name $name. Creating cache with default settings.")
            super.createNativeCaffeineCache(name)
        }
      }

      override def setCacheLoader(cacheLoader: CacheLoader[AnyRef, AnyRef]): Unit = {
        throw new UnsupportedOperationException("Cache loading not support by this manager implementation")
      }
    }
    val tileType = Type.valueOf(classOf[Tile])
    cacheManager.setCaffeine(cacheBuilderFrom(tileType))
    cacheManager
  }

  private def cacheBuilderFrom(tileType: Type): Caffeine[AnyRef, AnyRef] = {
    val expirationTime = tileType.getDescriptor.getPropertyDescriptor(Tile.PROPERTY_EXPIRATION_TIME).getDefaultValue.asInstanceOf[Integer].toLong
    val maxCacheEntries = tileType.getDescriptor.getPropertyDescriptor(Tile.PROPERTY_MAX_CACHE_ENTRIES).getDefaultValue.asInstanceOf[Integer].toLong
    val caffeineTileCacheBuilder = Caffeine.newBuilder()
      .expireAfterAccess(expirationTime, TimeUnit.SECONDS)
      .maximumSize(maxCacheEntries)
      .softValues()
      .recordStats()
    caffeineTileCacheBuilder
  }

}
