package com.xebialabs.xlrelease.plugins.dashboard.repository

import com.google.common.base.Preconditions.checkNotNull
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.Role
import com.xebialabs.xlrelease.plugins.dashboard.builder.DashboardBuilder.newDashboard
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.repository.Ids.{getParentId, releaseIdFrom}
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.ConfigurationReferencesSupport
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ReleaseConfigurationReferencePersistence
import com.xebialabs.xlrelease.service.{CiIdService, ReleaseService}
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper._
import grizzled.slf4j.Logging
import org.springframework.stereotype.Repository

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Repository
class ReleaseDashboardRepository(implicit val ciIdService: CiIdService,
                                 releaseService: ReleaseService,
                                 variableRepository: ReleaseVariableRepository,
                                 releaseExtensionsRepository: ReleaseExtensionsRepository,
                                 val configurationPersistence: ReleaseConfigurationReferencePersistence)
  extends DashboardRepository
    with ConfigurationReferencesSupport
    with Logging {

  releaseExtensionsRepository.registerGeneric(Dashboard.DASHBOARD_PREFIX)

  override def exists(dashboardId: String): Boolean = releaseExtensionsRepository.exists(dashboardId)

  override def search(parentId: String, title: String, principals: Iterable[String], roles: Iterable[Role], enforcePermissions: Boolean = true): Seq[Dashboard] =
    Seq.empty

  override def findDashboardById(dashboardId: String): Dashboard = {
    val dashboard = releaseExtensionsRepository.read[Dashboard](dashboardId).getOrElse(getDefaultDashboard(dashboardId))
    dashboard.setParentId(getParentId(dashboardId))
    dashboard
  }

  override def findTileById(tileId: String): Tile = findDashboardById(getParentId(tileId)).getTile(tileId)

  override def createDashboard(dashboard: Dashboard): Dashboard = createOrUpdateDashboard(dashboard)

  override def updateDashboard(dashboard: Dashboard): Dashboard = createOrUpdateDashboard(dashboard)

  override def deleteDashboard(dashboardId: String): Unit = {
    releaseExtensionsRepository.delete(dashboardId)
    val release = releaseService.findByCalendarToken(releaseIdFrom(dashboardId))
    updateConfigurationRefs(release)
  }

  private def createOrUpdateDashboard(dashboard: Dashboard): Dashboard = {
    checkNotNull(dashboard.getId)

    dashboard.setNewTileIds()

    interceptCreate(dashboard)

    releaseExtensionsRepository.createOrUpdate(dashboard)
    val release = releaseService.findById(releaseIdFrom(dashboard.getId))
    updateConfigurationRefs(release)
    val newVariables = scanAndBuildNewVariables(release, release, ciIdService).asScala.toSeq
    if (newVariables.nonEmpty) {
      variableRepository.create(newVariables, release)
    }

    dashboard
  }

  override def updateTile(tile: Tile): Tile = {
    val dashboardId = getParentId(tile.getId)

    createDashboard {
      releaseExtensionsRepository.read[Dashboard](dashboardId) match {
        case None =>
          val dashboard = getDefaultDashboard(dashboardId)
          dashboard.updateTile(tile)
          dashboard.updateTileIds()
          dashboard

        case Some(dashboard) =>
          dashboard.updateTile(tile)
          dashboard
      }
    }
    tile
  }

  override def getSecurityUid(dashboardId: String): Option[String] = None // current release dashboards do not have security restrictions

  private def getDefaultDashboard(dashboardId: String): Dashboard = {
    val release = releaseService.findByIdIncludingArchived(releaseIdFrom(dashboardId))
    if (release.isArchived) {
      release.getProperty[JList[ConfigurationItem]]("extensions").asScala.find(ci => Ids.getName(dashboardId) == Ids.getName(ci.getId)) match {
        case None =>
          createDefaultDashboard(dashboardId, release.isWorkflow)
        case Some(dashboard) =>
          dashboard.asInstanceOf[Dashboard]
      }
    } else {
      createDefaultDashboard(dashboardId, release.isWorkflow)
    }
  }

  private def createDefaultDashboard(dashboardId: String, isWorkflow: Boolean): Dashboard = {
    val dashboard = newDashboard.withId(dashboardId).build
    if (isWorkflow) {
      dashboard
    } else {
      dashboard.addDefaultTiles()
    }
  }

}
