package com.xebialabs.xlrelease.plugins.dashboard.repository

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.deployit.security.{PermissionEnforcer, Role}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.json.JsonUtils.DashboardJsonOps
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.plugins.dashboard.repository.sql.persistence.{DashboardPersistence, DashboardRow, DashboardSqlBuilder}
import com.xebialabs.xlrelease.repository.Ids.getParentId
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_DASHBOARD
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._

@Repository
class SqlDashboardRepository(implicit val ciIdService: CiIdService,
                             @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                             sqlRepositoryAdapter: SqlRepositoryAdapter,
                             permissionEnforcer: PermissionEnforcer,
                             dashboardPersistence: DashboardPersistence)
  extends Logging with DashboardRepository {

  override def exists(dashboardId: String): Boolean = dashboardPersistence.exists(dashboardId)

  def search(parentId: String, title: String, principals: Iterable[String], roles: Iterable[Role], enforcePermissions: Boolean = true): Seq[Dashboard] = {
    val query = new DashboardSqlBuilder()
      .select()
      .withTitleLike(title)
      .withParent(parentId)
      .ordered()

    if (enforcePermissions && !permissionEnforcer.isAdmin(principals.asJavaCollection, roles.toList.asJava)) {
      if (parentId == null) {
        query.withGlobalPermission(Option(VIEW_DASHBOARD), getAuthenticatedUserName, principals, roles.map(_.getId))
      } else {
        query.withFolderPermission(Option(VIEW_DASHBOARD), getAuthenticatedUserName, principals, roles.map(_.getId))
      }
    }

    dashboardPersistence.findByQuery(query.build()).map(toDashboard)
  }

  override def findDashboardById(dashboardId: String): Dashboard =
    dashboardPersistence.findById(dashboardId)
      .map(toDashboard)
      .getOrElse(throw new NotFoundException(s"Dashboard [$dashboardId] not found"))

  override def findTileById(tileId: String): Tile = findDashboardById(getParentId(tileId)).getTile(tileId)

  override def createDashboard(dashboard: Dashboard): Dashboard = {
    createDashboard(dashboard, true)
  }

  def createDashboard(dashboard: Dashboard, generateIds: Boolean): Dashboard = {
    if (generateIds) {
      dashboard.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Dashboard]), APPLICATIONS.getRootNodeName))
      dashboard.setNewTileIds()
    }

    interceptCreate(dashboard)

    dashboardPersistence.insert(dashboard)
    dashboard
  }

  override def updateDashboard(dashboard: Dashboard): Dashboard = {
    dashboard.setNewTileIds()
    interceptUpdate(dashboard)
    dashboardPersistence.update(dashboard)
    dashboard
  }

  override def deleteDashboard(dashboardId: String): Unit = {
    dashboardPersistence.delete(dashboardId)
  }

  override def updateTile(tile: Tile): Tile = {
    val dashboard = findDashboardById(getParentId(tile.getId))
    dashboard.updateTile(tile)
    updateDashboard(dashboard)
    tile
  }

  override def getSecurityUid(dashboardId: String): Option[String] = dashboardPersistence.getUid(dashboardId).map(_.toString)

  def tenantDashboardCount(tenantId: String): Int = {
    dashboardPersistence.tenantDashboardCount(tenantId)
  }

  private def toDashboard(row: DashboardRow): Dashboard = {
    val dashboard = sqlRepositoryAdapter.deserialize[Dashboard](row.content.withoutUnknownTiles).get
    if (dashboard.isFolderDashboard) {
      dashboard.setParentId(row.parentId)
    }
    dashboard
  }

}
