import { call, put } from 'redux-saga/effects';
import { registerSidebarProvider } from '@xlr-ui/app/features/custom-ui/sidebar.registry';
import { Folder } from '@xlr-ui/app/types';
import { getFolderOrWaitForIt } from '@xlr-ui/app/features/folders/ducks/folders.saga';
import { BreadcrumbItem, FolderPathService, MainNavigationItem, UiExtensionsService, XlrDashboardService } from '@xlr-ui/app/features/tasks/types/angular';
import getAngularService from '@xlr-ui/app/features/common/services/angular-accessor';
import { IStateService } from 'angular-ui-router';
import { navigation } from '@xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { DashboardSidebarProvider } from './xlr-dashboard-provider';
import { Dashboard } from './types';
import { IHttpResponse } from 'angular';
import { mapFolderTabs } from '@xlr-ui/app/features/folders/sidebar-provider/folder-sidebar-provider';
import { DEFAULT_FOLDER_TAB } from '@xlr-ui/app/features/folders/folder-constants';

const { setSidebarMenu, setLastBreadcrumb, updateBreadcrumbs, setBackItem } = navigation.actions;

export class FolderDashboardSidebarProvider extends DashboardSidebarProvider {
    isSupported(path: string): boolean {
        return path.startsWith('/folders/') && path.includes('/dashboards/Applications/Dashboard');
    }

    getDashboardsListPath(path: string) {
        return path.substring(0, path.lastIndexOf(this.dashboardsPrefix) + this.dashboardsPrefix.length);
    }

    getDashboardPath(path: string, isDetails: boolean) {
        return isDetails ? path.substring(0, path.lastIndexOf('/Tile')) : path;
    }

    getDashboardId(path: string, isInDashboardDetails: boolean) {
        const searchString = `${this.dashboardsPrefix}/`;
        const lastIndex = path.lastIndexOf(searchString);
        let dashboardId = path.substring(lastIndex + searchString.length);

        if (isInDashboardDetails) {
            dashboardId = dashboardId.replace(this.dashboardDetailsPattern, '');
        }
        return dashboardId;
    }

    *provide(path: string) {
        const isInDashboardDetails = this.checkIfInDashboardDetails(path);
        const dashboardId = this.getDashboardId(path, isInDashboardDetails);
        const xlrDashboardService: XlrDashboardService = yield call(getAngularService, 'xlrelease.XlrDashboardService');
        const { data: dashboard }: IHttpResponse<Dashboard> = yield call([xlrDashboardService, xlrDashboardService.loadConfiguration], dashboardId);
        const dashboardsListPath = this.getDashboardsListPath(path);
        const dashboardPath = this.getDashboardPath(path, isInDashboardDetails);

        const selectedFolder: Folder = yield call(getFolderOrWaitForIt);
        const uiExtensionsService: UiExtensionsService = yield call(getAngularService, 'UiExtensionsService');
        const folderTabs: MainNavigationItem[] = yield call(
            [uiExtensionsService, uiExtensionsService.getMenuItems],
            'xlrelease.sidebar.Folder',
            selectedFolder,
        );

        const $state: IStateService = yield call(getAngularService, '$state');
        const selectedPathSuffix = $state.params.page || DEFAULT_FOLDER_TAB;
        const folderSidebar = mapFolderTabs(folderTabs, selectedFolder, selectedPathSuffix);

        yield put(setSidebarMenu({ sidebarMenu: folderSidebar, withFolder: true }));

        const folderPathService: FolderPathService = yield call(getAngularService, 'FolderPathService');
        const parentFolders: Array<Folder> = yield call([folderPathService, folderPathService.getPathFromParentFolderId], selectedFolder.id);

        const folderBreadcrumbs: BreadcrumbItem[] = [];
        folderBreadcrumbs.push(
            ...parentFolders.map((f) => ({ text: f.title, href: `/folders/${f.id}/${selectedPathSuffix}` })),
            { text: 'Custom dashboards', href: dashboardsListPath },
            { text: dashboard.title, href: dashboardPath },
        );

        if (isInDashboardDetails) {
            folderBreadcrumbs.push({ text: 'Details' });
        }
        yield put(setLastBreadcrumb(undefined));
        // Clean potential back item leftovers
        yield put(setBackItem(null));
        yield put(updateBreadcrumbs(folderBreadcrumbs));
    }
}

registerSidebarProvider(new FolderDashboardSidebarProvider());
