package com.xebialabs.xlrelease.environments.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument}
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentLabelFilters
import com.xebialabs.xlrelease.domain.environments.EnvironmentLabel
import com.xebialabs.xlrelease.environments.repository.EnvironmentLabelRepository
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.util.regex.Pattern
import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class EnvironmentLabelService @Autowired()(environmentLabelRepository: EnvironmentLabelRepository) extends Logging {

  @Timed
  def searchLabels(environmentLabelFilters: EnvironmentLabelFilters): JList[EnvironmentLabel] = {
    environmentLabelRepository.search(environmentLabelFilters, defaultPage).asJava
  }

  @Timed
  def findLabelById(environmentLabelId: String): EnvironmentLabel = {
    environmentLabelRepository.findById(environmentLabelId)
  }

  @Timed
  def findByTitle(environmentName: String): EnvironmentLabel = {
    environmentLabelRepository.findByTitle(environmentName)
  }

  @Timed
  def createLabel(environmentLabel: EnvironmentLabel): EnvironmentLabel = {
    logger.debug("Creating a new environment label")
    validate(environmentLabel)
    environmentLabelRepository.create(environmentLabel)
  }

  @Timed
  def updateLabel(environmentLabel: EnvironmentLabel): EnvironmentLabel = {
    logger.debug(s"Updating environment label [$environmentLabel]")
    checkArgument(hasText(environmentLabel.getId), "ID is required")
    validate(environmentLabel)
    environmentLabelRepository.update(environmentLabel)
  }

  @Timed
  def deleteLabel(environmentLabelId: String): Unit = {
    logger.debug(s"Deleting environment label [$environmentLabelId]")
    environmentLabelRepository.delete(environmentLabelId)
  }

  private def validate(environmentLabel: EnvironmentLabel): Unit = {
    validateTitle(environmentLabel)
    validateColor(environmentLabel.getColor)
  }

  private def validateTitle(environmentLabel: EnvironmentLabel): Unit = {
    checkArgument(hasText(environmentLabel.getTitle), "Title cannot be blank")
  }

  private def validateColor(color: String): Unit = {
    if (!Pattern.compile("^#([A-Fa-f0-9]{6})$").matcher(color).matches()) {
      throw new IncorrectArgumentException(s"Invalid color pattern '$color'")
    }
  }
}
