import { SagaIterator } from 'redux-saga';
import { labelsState } from './labels.reducer';
import labelSaga, { _deleteLabel, _saveLabel, _searchLabels, _scheduleInitHeader, toastr } from './labels.saga';
import { actionTypes as schedulerActionTypes } from '../../../actions/scheduling-actions';
import { all, call, put, takeLatest, throttle } from 'redux-saga/effects';
import { httpDELETE, httpPOST, httpPUT } from '../../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { label, labels, newLabel } from '../__mocks__/environment-list.mocks';

const { searchLabels, saveLabel, setLabels, deleteLabel } = labelsState.actions;

describe('labels.saga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = labelSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeLatest(saveLabel, _saveLabel),
                takeLatest(deleteLabel, _deleteLabel),
                takeLatest(searchLabels, _searchLabels),
                throttle(200, schedulerActionTypes.SCHEDULE_INIT_HEADER_SUCCESS, _scheduleInitHeader),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('search labels', () => {
        it('should search and set labels', () => {
            const gen: SagaIterator = _searchLabels({
                payload: '',
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/labels/search', { title: '' }));
            expect(gen.next({ data: labels }).value).toStrictEqual(put(setLabels(labels)));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('save label', () => {
        it('should save new label', () => {
            const gen: SagaIterator = _saveLabel({
                payload: newLabel,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPOST, 'api/v1/environments/labels', newLabel));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Label [${newLabel.title}] created`));
            expect(gen.next().value).toStrictEqual(put(searchLabels('')));
            expect(gen.next().done).toBe(true);
        });

        it('should update label', () => {
            const gen: SagaIterator = _saveLabel({
                payload: label,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpPUT, `api/v1/environments/labels/${label.id}`, label));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Label [${label.title}] updated`));
            expect(gen.next().value).toStrictEqual(put(searchLabels('')));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('delete label', () => {
        it('should delete label', () => {
            const gen: SagaIterator = _deleteLabel({
                payload: label,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(call(httpDELETE, `api/v1/environments/labels/${label.id}`));
            expect(gen.next().value).toStrictEqual(call(toastr.success, `Label [${label.title}] deleted`));
            expect(gen.next().value).toStrictEqual(put(searchLabels('')));
            expect(gen.next().done).toBe(true);
        });
    });

    describe('schedule init header', () => {
        it('should set labels', () => {
            const gen: SagaIterator = _scheduleInitHeader({
                payload: {
                    applications: [],
                    folders: [],
                    stages: [],
                    labels,
                },
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(put(setLabels(labels)));
            expect(gen.next().done).toBeTruthy();
        });
    });
});
