package com.xebialabs.xlrelease.domain;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.*;
import org.apache.http.HttpHost;
import org.apache.http.ParseException;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.domain.variables.reference.PropertyUsagePoint;
import com.xebialabs.xlrelease.domain.variables.reference.UsagePoint;
import com.xebialabs.xlrelease.events.TaskStartOrRetryOperation;
import com.xebialabs.xlrelease.service.ExecuteTaskAction;
import com.xebialabs.xlrelease.variable.ValueWithInterpolation;

import static com.xebialabs.xlrelease.domain.GroovyScriptTask.GROOVY_ENGINE;
import static com.xebialabs.xlrelease.domain.ScriptTask.JYTHON_ENGINE;
import static com.xebialabs.xlrelease.script.UnsupportedScriptService.UNSUPPORTED_ENGINE;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAll;
import static com.xebialabs.xlrelease.variable.VariableHelper.replaceAllWithInterpolation;
import static java.net.URI.create;
import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.apache.http.HttpStatus.*;
import static org.springframework.util.StringUtils.isEmpty;

@PublicApiRef
@ShowOnlyPublicApiMembers
@Metadata(label = "Script: External Script", versioned = false, description = "Executes a script from an external source")
public class ExternalScriptTask extends ResolvableScriptTask {
    private static final Map<String, String> EXTENSION_ENGINE_MAP;

    private static final Set<Integer> COMMON_HTTP_STATUS_ERROR_CODES;

    @Property(description = "Script location.")
    private String url;

    @Property(description = "Username for basic authentication.", required = false)
    private String username;

    @Property(description = "Password for basic authentication.", required = false, password = true)
    private String password;

    @Property(description = "Language of the script. Allowed values are 'AUTODETECT', 'JYTHON' and 'GROOVY'.", required = false, defaultValue = "AUTODETECT")
    private ScriptEngine scriptEngine = ScriptEngine.AUTODETECT;

    static {
        EXTENSION_ENGINE_MAP = Map.of("py", JYTHON_ENGINE, "groovy", GROOVY_ENGINE);

        COMMON_HTTP_STATUS_ERROR_CODES = Set.of(SC_BAD_REQUEST, SC_UNAUTHORIZED, SC_FORBIDDEN, SC_NOT_FOUND, SC_CONFLICT,
                SC_INTERNAL_SERVER_ERROR, SC_BAD_GATEWAY, SC_SERVICE_UNAVAILABLE, SC_GATEWAY_TIMEOUT);
    }

    @Override
    protected Changes execute(String targetId, TaskStartOrRetryOperation operation) {
        return executeScript(targetId, operation, new ExecuteTaskAction(this));
    }

    @Override
    public String getEngine() {
        String engine;

        if (!isEmpty(scriptEngine) && scriptEngine != ScriptEngine.AUTODETECT) {
            return EXTENSION_ENGINE_MAP.values().stream()
                    .filter(e -> e.equals(scriptEngine.name().toLowerCase()))
                    .findFirst().orElse(UNSUPPORTED_ENGINE);
        }

        try {
            String extension = getExtension(create(url).getPath());
            engine = EXTENSION_ENGINE_MAP.getOrDefault(extension, UNSUPPORTED_ENGINE);
        } catch (IllegalArgumentException | NullPointerException e) {
            engine = UNSUPPORTED_ENGINE;
        }

        return engine;
    }

    @Override
    public String getScript() {
        String script = resolveScript();
        if (!isScriptInterpolationOff()) {
            script = replaceVariables(script);
        }

        return script;
    }

    private String replaceVariables(String script) {
        Release release = getRelease();
        Map<String, ValueWithInterpolation> variables = release.getAllVariableValuesAsStringsWithInterpolationInfo();
        return replaceAllWithInterpolation(script, variables, new HashSet<>(), false);
    }

    private String resolveScript() {
        String script;

        try (CloseableHttpClient httpClient = HttpClients.custom().build()) {
            HttpGet httpGet = new HttpGet(url);
            HttpClientContext httpClientContext = HttpClientContext.create();

            if (!isEmpty(username) && !isEmpty(password)) {
                addAuthentication(url, username, password, httpClientContext);
            }

            try (CloseableHttpResponse httpResponse = httpClient.execute(httpGet, httpClientContext)) {

                if (COMMON_HTTP_STATUS_ERROR_CODES.contains(httpResponse.getStatusLine().getStatusCode())) {
                    throw new IllegalStateException(String.format("Got status code %d while fetching the script from the url",
                            httpResponse.getStatusLine().getStatusCode()));
                }

                script = EntityUtils.toString(httpResponse.getEntity());
            } catch (IOException | ParseException e) {
                throw new IllegalStateException("Can't extract the script from the response", e);
            }
        } catch (IOException e) {
            throw new IllegalStateException("Can't fetch the script from the url", e);
        }

        return script;
    }

    private void addAuthentication(String url, String username, String password, HttpClientContext httpClientContext) {
        HttpHost targetHost;

        try {
            URL targetUrl = new URL(url);
            URIBuilder builder = new URIBuilder(url);
            final URI uri = builder.build();
            int port = uri.getPort();
            if (port == -1) {
                if (uri.getScheme().equalsIgnoreCase("https")) {
                    port = 443;
                } else {
                    port = 80;
                }
            }
            targetHost = new HttpHost(targetUrl.getHost(), port, targetUrl.getProtocol());
        } catch (MalformedURLException | URISyntaxException ex) {
            throw new IllegalArgumentException("Provided URL is not parsable", ex);
        }

        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(
                new AuthScope(targetHost.getHostName(), targetHost.getPort()),
                new UsernamePasswordCredentials(username, password));

        AuthCache authCache = new BasicAuthCache();
        BasicScheme basicScheme = new BasicScheme();
        authCache.put(targetHost, basicScheme);

        httpClientContext.setCredentialsProvider(credentialsProvider);
        httpClientContext.setAuthCache(authCache);
    }

    @Override
    public List<UsagePoint> getVariableUsages() {
        ArrayList<UsagePoint> usagePoints = new ArrayList<>(super.getVariableUsages());
        usagePoints.add(new PropertyUsagePoint(this, "url"));
        usagePoints.add(new PropertyUsagePoint(this, "username"));
        usagePoints.add(new PropertyUsagePoint(this, "password"));
        return usagePoints;
    }

    @Override
    public Set<String> freezeVariablesInCustomFields(Map<String, ValueWithInterpolation> variables, Map<String, String> passwordVariables,
                                                     @SuppressWarnings("unused") Changes changes, boolean freezeEvenIfUnresolved) {
        Set<String> unresolvedVariables = new HashSet<>();
        setUrl(replaceAllWithInterpolation(getUrl(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setUsername(replaceAllWithInterpolation(getUsername(), variables, unresolvedVariables, freezeEvenIfUnresolved));
        setPassword(replaceAll(getPassword(), passwordVariables, unresolvedVariables, freezeEvenIfUnresolved));
        return unresolvedVariables;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(final String url) {
        this.url = url;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(final String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(final String password) {
        this.password = password;
    }

    public ScriptEngine getScriptEngine() {
        return scriptEngine;
    }

    public void setScriptEngine(final ScriptEngine scriptEngine) {
        this.scriptEngine = scriptEngine;
    }

}
