import React from 'react';
import moment from 'moment';
import { DotIcon, DotTable, DotTooltip, DotTypography } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { primaryButton, secondaryButton } from '../../CommonCardTile/tile-buttons-row.mock';
import { ActiveReleaseTile, ActiveReleaseTileProps, HomeRelease } from './active-release-tile.component';
import { CommonActiveTile } from '../../CommonCardTile/common-active-tile.component';
import { getReleaseIdForUrl } from '../../TemplateTile/components/helpers/helpers';
import { DATE_FORMAT_DAY_FIRST, DATE_FORMAT_MONTH_FIRST } from '../../../../../../../../../../core/xlr-ui/app/js/locale/constants';
import * as angularAccessor from '../../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { getDaysHoursMinutesFromMS } from '../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/scheduling/helper';
import { formatReleaseDuration } from './helpers/helpers';
import { ReleaseStatus } from '../../../../../../../../../../core/xlr-ui/app/features/release-overview/release-status-label/release-status.component';
import { calculateDotDateFormat } from '../../../../../../../../../../core/xlr-ui/app/features/common/helper/date';

describe('ActiveReleaseTile', () => {
    const dispatch = jest.fn();

    const title = 'Releases';
    const releases: Array<HomeRelease> = [
        {
            releaseId: 'releaseId1',
            releaseTitle: 'Test Release',
            releaseStartDate: 1689771884388,
            releaseEndDate: 2689771884388,
            releaseStatus: 'IN_PROGRESS',
            folderName: 'test',
            folderId: 'folderId',
            folderPath: '/',
            lastEditedByUser: 1690187050870,
        },
        {
            releaseId: 'releaseId2',
            releaseTitle: 'Overdue release',
            releaseStartDate: 1,
            releaseEndDate: 1,
            releaseStatus: 'IN_PROGRESS',
            folderName: 'test',
            folderId: 'folderId',
            folderPath: '/',
            lastEditedByUser: 1,
        },
    ];

    const defaultProps: ActiveReleaseTileProps = {
        primaryButton,
        releases,
        secondaryButton,
        title,
    };

    const defaultState = {
        profile: { dateFormat: DATE_FORMAT_DAY_FIRST },
    };

    const mount = (props: ActiveReleaseTileProps = defaultProps, state = defaultState) => {
        return mountWithStoreAndTheme(<ActiveReleaseTile {...props} />, dispatch, state);
    };

    const getCommonActiveTile = (wrapper: ReactWrapper) => wrapper.find(CommonActiveTile);
    const getTable = (wrapper: ReactWrapper) => wrapper.find(DotTable);
    const getRowTitleData = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-release-title');
    const getRowFolderData = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotTypography) && node.props().className === 'release-folder-name');
    const getOverdueIcon = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotIcon) && node.props().iconId === 'delay');
    const getReleaseStatus = (wrapper: ReactWrapper) => wrapper.find(ReleaseStatus);
    const getEndDateTooltip = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotTooltip) && node.props()['data-testid'] === 'end-date-tooltip');
    const getRowDuration = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'row-release-duration');

    beforeEach(() => {
        const getAngularServiceSpy = jest.spyOn(angularAccessor, 'default') as unknown as jest.SpyInstance;
        const dateFilterWrapper = (_filter: string) => (date: string, angularJsFormat: string) => moment(date).format(angularJsFormat);
        getAngularServiceSpy.mockReturnValue(dateFilterWrapper);
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render with correct props', () => {
        const wrapper = mount();

        const commonActiveTileProps = getCommonActiveTile(wrapper).props();
        expect(commonActiveTileProps.avatarIcon).toBe('release');
        expect(commonActiveTileProps.primaryButton).toBe(primaryButton);
        expect(commonActiveTileProps.secondaryButton).toBe(secondaryButton);
        expect(commonActiveTileProps.subtitle).toBe('Last updated - Most recent 5');
        expect(commonActiveTileProps.title).toBe(title);
    });

    it('should render main content correctly', () => {
        const wrapper = mount();

        const columns = [
            { id: 'releases', label: 'Release', truncate: true },
            { id: 'status', label: 'Status', width: '140px' },
            { id: 'endDate', label: 'End date', width: '160px' },
            { id: 'duration', label: 'Duration', width: '140px' },
        ];

        const tableProps = getTable(wrapper).props();
        expect(tableProps.className).toBe('active-release-table');
        expect(tableProps.columns).toStrictEqual(columns);
        expect(tableProps.sortable).toBe(false);
    });

    it('should have correct table data', () => {
        const wrapper = mount();

        const rowTitle = getRowTitleData(wrapper).at(0);
        expect(rowTitle.props().variant).toBe('body1');
        expect(rowTitle).toHaveText(releases[0].releaseTitle);

        const rowFolderName = getRowFolderData(wrapper).at(0);
        expect(rowFolderName.props().variant).toBe('body2');
        expect(rowFolderName).toHaveText(releases[0].folderName);

        const releaseStatusComponent = getReleaseStatus(wrapper).at(0);
        expect(releaseStatusComponent).toExist();
        const releaseStatusComponentProps = releaseStatusComponent.props();
        expect(releaseStatusComponentProps.status).toBe(releases[0].releaseStatus);

        const endDateTooltip = getEndDateTooltip(wrapper).at(0);
        const endDateTooltipProps = endDateTooltip.props();
        expect(endDateTooltipProps.title).toBe(moment(releases[0].releaseEndDate).format(calculateDotDateFormat(DATE_FORMAT_DAY_FIRST)));
        expect(endDateTooltipProps.placement).toBe('bottom-start');
        const endDate = endDateTooltip.find(DotTypography);
        expect(endDate.props().variant).toBe('body1');
        expect(endDate).not.toHaveClassName('overdue');
        expect(endDate).toHaveText(moment(releases[0].releaseEndDate).fromNow());

        const [durationDays, durationHours, durationMinutes] = getDaysHoursMinutesFromMS(releases[0].releaseEndDate - releases[0].releaseStartDate);

        const duration = getRowDuration(wrapper).at(0);
        expect(duration.props().variant).toBe('body1');
        expect(duration).toHaveText(formatReleaseDuration(durationDays, durationHours, durationMinutes));
    });

    it('should display overdue information', () => {
        const wrapper = mount();
        expect(getOverdueIcon(wrapper)).toExist();
        const endDateTooltip = getEndDateTooltip(wrapper).at(1);
        const endDate = endDateTooltip.find(DotTypography);
        expect(endDate).toHaveClassName('overdue');
    });

    it('should handle on click for table row', () => {
        const wrapper = mount();
        const table = getTable(wrapper);
        table.invoke('onRowClick')?.({} as never, getReleaseIdForUrl(releases[0]));
        expect(window.location.href).toStrictEqual(`http://localhost/#/releases/${getReleaseIdForUrl(releases[0])}`);
    });

    it('should display correct date format based on profile settings', () => {
        const wrapper = mount(defaultProps, {
            ...defaultState,
            profile: { ...defaultState.profile, dateFormat: DATE_FORMAT_MONTH_FIRST },
        });
        expect(getEndDateTooltip(wrapper).at(0).props().title).toBe(moment(releases[0].releaseEndDate).format(calculateDotDateFormat(DATE_FORMAT_MONTH_FIRST)));
    });
});
