import React, { ReactElement } from 'react';
import { ReactWrapper } from 'enzyme';
import { DotAutoComplete, AutoCompleteProps, AutoCompleteOption, DotInputText } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { stringPropertyMock } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/property.mock';
import { taskMock } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/task.mock';
import { StageWithVariables } from './stage.component';
import { AUTOCOMPLETE_MAX_HEIGHT } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/common/constants';
import { RootState } from '../../../../../../../../../../../core/xlr-ui/app/js/store.types';
import { PropertiesEnum, TaskInputCommonProps } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/config/components/types';

describe('StageWithVariables component', () => {
    const defaultState = {
        taskDrawer: {
            variables: [{ key: 'var' }],
        },
        deliveryTasks: {
            deliveriesOrPatterns: [],
            deliveryOrPattern: null,
            patterns: [],
        },
    };

    const deliveryStages = [
        {
            id: 'DeliveryId/Stage1',
            title: 'Stage one',
        },
        {
            id: 'DeliveryId/Stage2',
            title: 'Stage two',
        },
    ];

    const patternStages = [
        {
            id: 'PatternId/Stage1',
            title: 'Stage one',
        },
        {
            id: 'PatternId/Stage2',
            title: 'Stage two',
        },
    ];

    const defaultTask = {
        ...taskMock,
        inputProperties: {
            deliveryId: 'DeliveryId',
            stage: 'DeliveryId/Stage1',
        },
    };

    const onChange = jest.fn();
    const dispatch = jest.fn();
    let wrapper: ReactWrapper;

    const defaultProps: TaskInputCommonProps = {
        disabled: false,
        expanded: false,
        onChange,
        propertiesEnum: PropertiesEnum.InputProperties,
        property: stringPropertyMock,
        task: defaultTask,
    };

    const mount = (component: ReactElement, state: RootState = defaultState) => {
        wrapper = mountWithStoreAndTheme(component, dispatch, state);
    };

    const mountStageWithVariables = (props: TaskInputCommonProps = defaultProps, state: RootState = defaultState) =>
        mount(<StageWithVariables {...props} />, state);

    const getDotAutoComplete = () => wrapper.find(DotAutoComplete);
    const getDotInputText = () => wrapper.find(DotInputText);

    const verifyCommonAutocompleteProps = (autocompleteProps: AutoCompleteProps<AutoCompleteOption>) => {
        expect(autocompleteProps.readOnly).toBe(false);
        expect(autocompleteProps.group).toBe(false);
        expect(autocompleteProps.helperText).toBe(stringPropertyMock.description);
        expect(autocompleteProps.inputId).toBe(stringPropertyMock.name);
        expect(autocompleteProps.label).toBe(stringPropertyMock.label);
        expect(autocompleteProps.maxHeight).toBe(AUTOCOMPLETE_MAX_HEIGHT);
        expect(autocompleteProps.multiple).toBe(false);
        expect(autocompleteProps.persistentLabel).toBe(true);
        expect(autocompleteProps.required).toBe(stringPropertyMock.required);
    };

    beforeEach(() => {
        dispatch.mockReset();
        onChange.mockReset();
    });

    it('should have the proper props when stage belongs to a delivery', () => {
        const state = {
            taskDrawer: {
                variables: [{ key: 'var' }],
            },
            deliveryTasks: {
                deliveriesOrPatterns: [],
                patterns: [],
                deliveryOrPattern: {
                    id: 'DeliveryId',
                    title: 'My Delivery',
                    stages: deliveryStages,
                },
            },
        };

        mountStageWithVariables(defaultProps, state);
        const autocompleteProps = getDotAutoComplete().props();
        verifyCommonAutocompleteProps(autocompleteProps);
        expect(autocompleteProps.value).toStrictEqual({
            id: 'DeliveryId/Stage1',
            title: 'Stage one',
        });
        expect(autocompleteProps.options).toStrictEqual(deliveryStages);
    });

    it('should have the proper props when stage belongs to a pattern', () => {
        const state = {
            taskDrawer: {
                variables: [{ key: 'var' }],
            },
            deliveryTasks: {
                deliveriesOrPatterns: [],
                patterns: [
                    {
                        id: 'PatternId',
                        title: 'My Pattern',
                        stages: patternStages,
                    },
                ],
                deliveryOrPattern: null,
            },
        };

        const task = {
            ...taskMock,
            inputProperties: {
                deliveryId: '${var}',
                stage: 'PatternId/Stage1',
                patternId: 'PatternId',
            },
        };

        mountStageWithVariables({ ...defaultProps, task }, state);
        const autocompleteProps = getDotAutoComplete().props();
        verifyCommonAutocompleteProps(autocompleteProps);
        expect(autocompleteProps.value).toStrictEqual({
            id: 'PatternId/Stage1',
            title: 'Stage one',
        });
        expect(autocompleteProps.options).toStrictEqual(patternStages);
    });

    it('should not show stages if deliveryId has no value', () => {
        const taskWithNoDeliveryId = { ...taskMock, inputProperties: { deliveryId: null } };
        mountStageWithVariables({ ...defaultProps, task: taskWithNoDeliveryId });
        expect(onChange).not.toHaveBeenCalled();
        expect(dispatch).not.toHaveBeenCalled();
        expect(getDotAutoComplete().props().options).toStrictEqual([]);
    });

    it('should show the value on the input text if is not a stageID', () => {
        const taskWithStageName = { ...taskMock, inputProperties: { deliveryId: null, stage: 'My Stage' } };
        mountStageWithVariables({ ...defaultProps, task: taskWithStageName });
        expect(onChange).not.toHaveBeenCalled();
        expect(dispatch).not.toHaveBeenCalled();
        expect(getDotInputText().props().value).toStrictEqual('My Stage');
    });
});
