package com.xebialabs.xlrelease.delivery.repository.sql

import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService
import com.xebialabs.xlrelease.api.v1.forms.{AbstractDeliveryFilters, DeliveryFilters, DeliveryOrderMode}
import com.xebialabs.xlrelease.api.v1.views.DeliveryFlowReleaseInfo
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.repository.sql.persistence._
import com.xebialabs.xlrelease.delivery.service.DeliveryDetailsDecorator.DELIVERY_DETAILS
import com.xebialabs.xlrelease.domain.delivery.{Delivery, DeliveryStatus, TrackedItem}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.repository.sql.{InterceptedRepository, SqlRepositoryAdapter}
import com.xebialabs.xlrelease.security.XLReleasePermissions.AUDIT_ALL
import com.xebialabs.xlrelease.utils.{Diff, TenantContext}
import com.xebialabs.xlrelease.utils.ScopedTokenPermissionValidator.checkPermissionSupportedinScopedToken
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Repository

import java.util
import scala.jdk.CollectionConverters._

@IsTransactional
@Repository
class SqlDeliveryRepository(deliveryPersistence: DeliveryPersistence,
                            deliveryMemberPersistence: DeliveryMemberPersistence,
                            trackedItemPersistence: TrackedItemPersistence,
                            sqlRepositoryAdapter: SqlRepositoryAdapter,
                            implicit val permissionEnforcer: PermissionEnforcer,
                            @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                            val decoratorService: InternalMetadataDecoratorService)
  extends DeliveryRepository
    with InterceptedRepository[Delivery] {

  @IsReadOnly
  override def read(deliveryId: String): Delivery =
    mapToDelivery(deliveryPersistence.findById(deliveryId))
      .getOrElse(throw new LogFriendlyNotFoundException(s"Release delivery $deliveryId not found"))

  @IsReadOnly
  override def getByIdOrTitle(deliveryIdOrTitle: String): Delivery =
    mapToDelivery(deliveryPersistence.findById(deliveryIdOrTitle)
      .orElse(deliveryPersistence.findByTitle(deliveryIdOrTitle)))
      .getOrElse(throw new LogFriendlyNotFoundException(s"Release delivery $deliveryIdOrTitle not found"))

  override def create(delivery: Delivery): Unit = {
    delivery.setTenantId(TenantContext.getTenant())
    interceptCreate(delivery)
    val deliveryUid = deliveryPersistence.insert(delivery)
    deliveryMemberPersistence.insertMembers(deliveryUid, delivery.getReleaseIds.asScala.toSet)
    trackedItemPersistence.insertItems(deliveryUid, delivery.getTrackedItems.asScala.toSet)
  }

  override def update(delivery: Delivery): Unit = {
    val originalDelivery = read(delivery.getId)
    val deliveryUid = deliveryPersistence.getUid(delivery.getId).get

    // Synchronize items
    delivery.getTrackedItems.asScala
    val itemDiff = Diff.applyWithKeyMappingAndComparator(
      originalDelivery.getTrackedItems.asScala,
      delivery.getTrackedItems.asScala
    )(_.getId, areItemsInTableEqual)

    itemDiff.updatedValues.foreach { item =>
      trackedItemPersistence.updateTrackedItem(item.getId, item)
    }
    trackedItemPersistence.insertItems(deliveryUid, itemDiff.newValues.toSet)
    trackedItemPersistence.deleteItems(deliveryUid, itemDiff.deletedKeys)

    // Synchronize release ids
    delivery.computeReleasesFromTrackedItems()
    val membersDiff = Diff(
      originalDelivery.getReleaseIds.asScala,
      delivery.getReleaseIds.asScala
    )
    deliveryMemberPersistence.deleteMembers(deliveryUid, membersDiff.deletedKeys)
    deliveryMemberPersistence.insertMembers(deliveryUid, membersDiff.newKeys)

    // Update JSON
    deliveryPersistence.update(delivery)
  }

  @IsReadOnly
  override def exists(deliveryId: String): Boolean = deliveryPersistence.exists(deliveryId)

  override def delete(deliveryId: String): Unit = deliveryPersistence.delete(deliveryId)

  @IsReadOnly
  override def search(filters: AbstractDeliveryFilters,
                      page: Page,
                      orderBy: DeliveryOrderMode,
                      principals: Iterable[String],
                      roleIds: Iterable[String],
                      anyOfPermissions: Seq[Permission],
                      enforcePermissions: Boolean = true): Seq[Delivery] = {
    val permissions = if (!enforcePermissions || (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL))) {
      Seq.empty[Permission]
    } else {
      checkPermissionSupportedinScopedToken(anyOfPermissions: _*)
      anyOfPermissions
    }
    val deliveryRows = if (filters.hasStatuses) {
      val sqlWithParams = new DeliverySqlBuilder()
        .select()
        .withTitleLike(filters.getTitle, filters.isStrictTitleMatch)
        .withTrackedItemTitleLike(filters.getTrackedItemTitle, filters.isStrictTrackedItemTitleMatch)
        .withOneOfStatuses(Option(filters.getStatuses).map(_.asScala.toSeq).getOrElse(Seq.empty))
        .withFolder(filters.getFolderId)
        .withPermissions(permissions, principals, roleIds
        )
        .withOriginPatternId(
          filters match {
            case filters1: DeliveryFilters => filters1.getOriginPatternId
            case _ => null
          }
        )
        .orderBy(orderBy)
        .limitAndOffset(page.resultsPerPage, page.resultsPerPage * page.page)
        .build()
      deliveryPersistence.findByQuery(sqlWithParams)
    } else {
      Seq.empty
    }

    val deliveries = toDeliveries(deliveryRows)
    if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
      deliveries
    } else {
      decorateWithEffectiveSecurityAndPatternDetails(deliveries)
    }
  }

  @IsReadOnly
  override def searchIds(filters: AbstractDeliveryFilters): Seq[String] = {
    val sqlWithParams = new DeliverySqlBuilder()
      .select()
      .withTitle(filters.getTitle)
      .withOneOfStatuses(Option(filters.getStatuses).map(_.asScala.toSeq).getOrElse(Seq.empty))
      .withFolder(filters.getFolderId)
      .build()
    deliveryPersistence.findByQuery(sqlWithParams).map(_.domainId)
  }

  override def findDeliveriesReferencingReleaseId(releaseId: String): Seq[String] = deliveryMemberPersistence.findDeliveriesReferencingReleaseId(releaseId)

  @IsReadOnly
  override def findActiveDeliveriesReferencingFolderId(folderId: String): Seq[CiIdWithTitle] =
    deliveryMemberPersistence.findActiveDeliveriesReferencingFolderId(folderId)

  @IsReadOnly
  override def findReleasesByDeliveryId(deliveryId: String): Seq[DeliveryFlowReleaseInfo] =
    deliveryPersistence.findReleasesByDeliveryId(deliveryId)
      .map(row => new DeliveryFlowReleaseInfo(row.id, row.title, row.status, row.startDate, row.endDate, false))

  private def toDeliveries(rows: Seq[DeliveryRow]): Seq[Delivery] = {
    val releaseIdsByDelivery = deliveryMemberPersistence.findMembersByDeliveryUids(rows.map(_.ciUid))
    rows.map(row => toDelivery(row, releaseIdsByDelivery.getOrElse(row.ciUid, Seq.empty)))
  }

  private def toDelivery(row: DeliveryRow, newReleaseIds: Seq[String]): Delivery = {
    val deliveryOption = sqlRepositoryAdapter.deserialize[Delivery](row.content)
    deliveryOption match {
      case Some(delivery) =>
        delivery.setFolderId(row.folderId.absolute)
        val mergedReleaseIds = delivery.getReleaseIds.asScala.map { oldId =>
          newReleaseIds.find(newId => getName(newId) == getName(oldId)).getOrElse(oldId)
        }
        delivery.setReleaseIds(new util.HashSet(mergedReleaseIds.asJava))
        delivery.setTenantId(row.tenantId)
        delivery
      case None =>
        throw new LogFriendlyNotFoundException(s"Error reading release delivery ${row.ciUid}, see logs for more details")
    }
  }

  @IsReadOnly
  override def findFolderId(deliveryId: String): String = deliveryPersistence.findFolderId(deliveryId).absolute

  @IsReadOnly
  override def findPatternByTitle(title: String): Option[Delivery] = {
    mapToDelivery(deliveryPersistence.findPatternByTitle(title))
  }

  @IsReadOnly
  override def tenantPatternCount(tenantId: String): Integer = {
    deliveryPersistence.tenantDeliveryCount(tenantId, DeliveryStatus.TEMPLATE)
  }

  @IsReadOnly
  override def tenantDeliveryCount(tenantId: String): Integer = {
    deliveryPersistence.tenantDeliveryCount(tenantId, DeliveryStatus.IN_PROGRESS)
  }

  private def decorateWithEffectiveSecurityAndPatternDetails(deliveries: Seq[Delivery]): Seq[Delivery] = deliveries.map(delivery => {
    decoratorService.decorate(delivery, Seq(EFFECTIVE_SECURITY, DELIVERY_DETAILS).asJava)
    delivery
  })

  private def mapToDelivery(deliveryRowOpt: Option[DeliveryRow]) = {
    deliveryRowOpt
      .map(row => toDelivery(row, deliveryMemberPersistence.findMembersByDeliveryUid(row.ciUid)))
      .map(delivery => decorateWithEffectiveSecurityAndPatternDetails(Seq(delivery)).head)
  }

  private def areItemsInTableEqual(oldItem: TrackedItem, newItem: TrackedItem): Boolean = {
    oldItem.getTitle == newItem.getTitle &&
      oldItem.isDescoped == newItem.isDescoped &&
      oldItem.getModifiedDate == newItem.getModifiedDate
  }
}
