package com.xebialabs.xlrelease.actors

import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, HOT_STANDBY, STANDALONE}
import com.xebialabs.xlrelease.actors.initializer.ActorInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.ReleaseGroupService
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorRef, PoisonPill, Props}
import org.apache.pekko.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings, ClusterSingletonProxy, ClusterSingletonProxySettings}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

import scala.concurrent.{Await, Promise}

trait ReleaseGroupActorInitializer extends ActorInitializer with Logging {
  def actorProps(releaseGroupService: ReleaseGroupService): Props = ReleaseGroupProcessingActor.props(releaseGroupService)
}

@Component
class ReleaseGroupActorHolder @Autowired()(xlrConfig: XlrConfig) {

  private val actorPromise = Promise[ActorRef]()

  def actorRef(): ActorRef = {
    Await.result(actorPromise.future, xlrConfig.timeouts.systemInitialization)
  }

  def resolveActorRef(actorRef: ActorRef): Unit = actorPromise.success(actorRef)
}

@Component
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredReleaseGroupActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                            releaseGroupService: ReleaseGroupService,
                                                            releaseGroupActorHolder: ReleaseGroupActorHolder) extends ReleaseGroupActorInitializer {

  lazy val releaseGroupActor: ActorRef = systemHolder.actorSystem.actorOf(actorProps(releaseGroupService))

  override def initialize(): Unit = {
    logger.debug("Initializing non-clustered release group actor...")
    releaseGroupActorHolder.resolveActorRef(releaseGroupActor)
    releaseGroupActor
  }
}

@Component
@Profile(Array(FULL))
class ClusteredReleaseGroupActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                         releaseGroupService: ReleaseGroupService,
                                                         releaseGroupActorHolder: ReleaseGroupActorHolder) extends ReleaseGroupActorInitializer {

  lazy val releaseGroupActor: ActorRef = clusteredArchivingActor(actorProps(releaseGroupService))

  override def initialize(): Unit = {
    logger.debug("Initializing clustered release group actor...")
    releaseGroupActorHolder.resolveActorRef(releaseGroupActor)
    releaseGroupActor
  }

  private[actors] def clusteredArchivingActor(actorProps: Props): ActorRef = {
    systemHolder.actorSystem.actorOf(ClusterSingletonManager.props(
      singletonProps = actorProps,
      terminationMessage = PoisonPill,
      settings = ClusterSingletonManagerSettings(systemHolder.actorSystem)),
      name = ReleaseGroupProcessingActor.name)

    systemHolder.actorSystem.actorOf(ClusterSingletonProxy.props(
      singletonManagerPath = s"user/${ReleaseGroupProcessingActor.name}",
      settings = ClusterSingletonProxySettings(systemHolder.actorSystem)),
      name = s"${ReleaseGroupProcessingActor.name}Proxy")
  }
}


