package com.xebialabs.xlrelease.reports.job.impl.cleanup

import akka.actor.ActorRef
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.configuration.AuditReportSettings
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.reports.job.api.ReportingEngineService.ReportJobId
import com.xebialabs.xlrelease.reports.job.domain.ReportJobStatus
import com.xebialabs.xlrelease.reports.job.domain.ReportJobStatus.{ABORTED, COMPLETED, FAILED}
import com.xebialabs.xlrelease.reports.job.events.ReportJobDeletedEvent
import com.xebialabs.xlrelease.reports.job.impl.ReportJobDelegateActor.Enqueue
import com.xebialabs.xlrelease.reports.job.impl.{GlobalReportStorage, ReportJobDelegateActorHolder, ReportJobInstanceFactory}
import com.xebialabs.xlrelease.reports.job.repository.{ReportJobFilters, ReportJobRepository}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import grizzled.slf4j.Logging
import org.springframework.data.domain.PageRequest
import org.threeten.extra.Interval

import java.time.{Instant, LocalDateTime, ZoneId}
import java.util.Date
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.Try

class ReportCleanerOperations(reportJobRepository: ReportJobRepository,
                              reportJobInstanceFactory: ReportJobInstanceFactory,
                              configurationRepository: ConfigurationRepository,
                              globalReportStorage: GlobalReportStorage,
                              reportJobDelegateActorHolder: ReportJobDelegateActorHolder,
                              eventBus: XLReleaseEventBus
                             )
  extends Logging {

  lazy val reportJobDelegateActor: ActorRef = reportJobDelegateActorHolder.actorRef()

  val initializationTime: Date = Date.from(Instant.now())

  def cleanup(): Unit = {
    statusCleanup()
    storageCleanup()
  }

  def nodeCleanup(node: String): Unit = {
    logger.debug(s"Report status cleanup for node $node started")
    if (Try(reportJobRepository.startedNodeJobIds(node).foreach(failNodeJob)).isFailure) {
      logger.warn(s"Report status cleanup failed for jobs on node '$node'")
    }
    if (Try(reportJobRepository.submittedNodeJobIds(node).foreach(requeue)).isFailure) {
      logger.warn(s"Requeue failed for jobs that were submitted on node '$node'")
    }
    logger.debug(s"Report status cleanup for node $node ended")
  }

  private[impl] def statusCleanup(): Unit = {
    logger.debug("Report status cleanup started")
    if (Try(reportJobRepository.startedStaleJobIds(initializationTime).foreach(failStaleJob)).isFailure) {
      logger.warn("Report status cleanup failed for stale jobs")
    }
    if (Try(reportJobRepository.submittedStaleJobIds(initializationTime).foreach(requeue)).isFailure) {
      logger.warn("Requeue failed for previously submitted jobs")
    }
    logger.debug("Report status cleanup ended")
  }

  private def storageCleanup(): Unit = {
    logger.debug("Report storage cleanup started")
    val reportSettings = configurationRepository.read[AuditReportSettings](AuditReportSettings.AUDIT_REPORT_SETTINGS_ID)
    val filter = new ReportJobFilters
    val retentionCutOffDate = LocalDateTime.now().minusDays(reportSettings.getReportsRetentionPeriod.toLong).atZone(ZoneId.systemDefault())
    filter.endTime = Interval.of(Instant.MIN, retentionCutOffDate.toInstant)
    filter.statuses = List(ABORTED.name(), FAILED.name(), COMPLETED.name()).asJava
    val result = reportJobRepository.query(filter, PageRequest.of(0, Int.MaxValue))
    val events = ListBuffer.empty[XLReleaseEvent]
    result.get().forEach {
      reportJob =>
        try {
          if (globalReportStorage.cleanup(reportJob)) {
            reportJob.status = ReportJobStatus.DELETED
            events += ReportJobDeletedEvent(reportJobRepository.update(reportJob))
          }
        } catch {
          case e: Exception => logger.warn(e.getMessage)
        }
    }
    events.foreach(eventBus.publish)
    logger.debug("Report storage cleanup finished")
  }


  private def failNodeJob(jobId: ReportJobId): Unit = {
    fail(jobId, s"Job '$jobId' marked as '${
      ReportJobStatus.FAILED
    }' because it was executing on a node that was removed from XLR cluster.")
  }


  private def requeue(jobId: ReportJobId): Unit = {
    val reportJobInstance = reportJobRepository.findByJobId(jobId)
      .map(reportJobInstanceFactory.create)
      .getOrElse(throw new NotFoundException(s"Job with id '$jobId' not found."))
    reportJobInstance.enqueue()
    reportJobDelegateActor ! Enqueue(jobId, reportJobInstance)
  }

  private def failStaleJob(jobId: ReportJobId): Unit = fail(jobId, s"Job '$jobId' is a stale report job that should be ${
    ReportJobStatus.FAILED
  }")


  private def fail(jobId: ReportJobId, msg: String): Unit = {
    for {
      reportJob <- reportJobRepository.findByJobId(jobId)
      jobInstance <- Some(reportJobInstanceFactory.create(reportJob))
    } jobInstance.fail(new IllegalStateException(msg))
  }

}
