package com.xebialabs.xlrelease.reports.repository.deployment

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.udm.reporting.DeploymentRecord
import com.xebialabs.xlrelease.domain.utils.FullReleaseId
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.reports.repository.CommonRowAndParamsMapper._
import com.xebialabs.xlrelease.reports.repository.deployment.DeploymentRecordArchiveRepository._
import com.xebialabs.xlrelease.reports.repository.{ArchivedTasksRepository, RepositoryExceptionUtils, SearchRecordsSupport}
import com.xebialabs.xlrelease.repository.FacetRepository.SpecializedFacetRepository
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.{FacetRepository, Ids}
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.util.Failure

object DeploymentRecordArchiveRepository {

  import DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE._

  private lazy val ALL_FIELDS =
    s"""| ${commonFieldNames(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE, "b")},
        | t.$REPORT_TASKS_ID_COLUMN,
        | b.$DEPLOYMENT_TASK,
        | b.$DEPLOYMENT_TASK_URL,
        | b.$ENVIRONMENT_NAME,
        | b.$APPLICATION_NAME,
        | b.$VERSION,
        | b.$STATUS""".stripMargin

  private lazy val SELECT_ALL_DEPLOYMENT_RECORDS =
    s"""SELECT
       | $ALL_FIELDS
       |FROM $TABLE b
       |JOIN $REPORT_TASKS_TABLE_NAME t ON t.$REPORT_TASKS_UID_COLUMN = b.$TASK_UID""".stripMargin

  private lazy val SELECT_DEPLOYMENT_RECORD_BY_ID_STMT: String =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |WHERE b.$RECORD_ID = :recordId""".stripMargin

  private lazy val SELECT_DEPLOYMENT_RECORDS_BY_TASK_ID_STMT =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |WHERE t.$REPORT_TASKS_ID_COLUMN = :taskId""".stripMargin

  private lazy val SELECT_DEPLOYMENT_RECORDS_BY_RELEASE_ID_STMT: String =
    s"""|$SELECT_ALL_DEPLOYMENT_RECORDS
        |JOIN $REPORT_RELEASES_TABLE_NAME r ON r.$REPORT_RELEASES_ID_COLUMN = t.$REPORT_TASKS_RELEASEID_COLUMN
        |WHERE r.$REPORT_RELEASES_ID_COLUMN = :releaseId AND r.$REPORT_RELEASES_PRE_ARCHIVED <> 1""".stripMargin

  private lazy val EXISTS_DEPLOYMENT_RECORD_BY_ID: String =
    s"""SELECT COUNT($RECORD_ID) FROM $TABLE WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val INSERT_DEPLOYMENT_RECORD_STMT =
    s"""| INSERT INTO ${TABLE} (
        | ${commonFieldNames(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE)},
        | $TASK_UID,
        | $DEPLOYMENT_TASK,
        | $DEPLOYMENT_TASK_URL,
        | $ENVIRONMENT_NAME,
        | $APPLICATION_NAME,
        | $VERSION,
        | $STATUS
        |) VALUES (
        | ${commonParameterNames()},
        | :taskUid,
        | :deploymentTask,
        | :deploymentTaskUrl,
        | :environmentName,
        | :applicationName,
        | :version,
        | :status
        |)""".stripMargin

  private lazy val DELETE_DEPLOYMENT_RECORD_STMT: String =
    s"""|DELETE FROM $TABLE
        |WHERE $RECORD_ID = :recordId""".stripMargin

  private lazy val FIND_ALL_APPLICATION_NAMES =
    s"""SELECT DISTINCT $APPLICATION_NAME
       |FROM $TABLE""".stripMargin

  private lazy val FIND_ALL_ENVIRONMENT_NAMES =
    s"""SELECT DISTINCT $ENVIRONMENT_NAME
       |FROM $TABLE""".stripMargin

}

class DeploymentRecordArchiveRepository(archivedTasksRepository: ArchivedTasksRepository,
                                        val ciIdService: CiIdService,
                                        val jdbcTemplate: JdbcTemplate,
                                        val dialect: Dialect)
  extends SpecializedFacetRepository(Type.valueOf(classOf[DeploymentRecord]))
    with FacetRepository.ForArchive
    with PersistenceSupport
    with SearchRecordsSupport[DeploymentRecord]
    with RepositoryExceptionUtils
    with Logging {

  override def get(recordId: String, recordType: Option[Type]): Facet = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(SELECT_DEPLOYMENT_RECORD_BY_ID_STMT, params("recordId" -> getName(recordId.normalized)), deploymentRowMapper)
      .headOption.getOrElse(throw new NotFoundException(s"Record '$recordId' not found."))
  }

  override protected def doCreate(facet: Facet): Facet = {
    logger.debug("Creating Deployment record in archive")
    requiresInstanceOfSupportedType(facet.getType)
    archivedTasksRepository.taskUidById(shortenId(facet.getTargetId)).map { taskUid =>
      val deploymentFacet = facet.asInstanceOf[DeploymentRecord]
      sqlInsert(INSERT_DEPLOYMENT_RECORD_STMT, asQueryParameters(deploymentFacet, taskUid))
      facet
    }.recoverWith {
      case _: NotFoundException =>
        Failure(targetTaskNotFoundException(facet.getId, facet.getTargetId))
    }.get
  }

  override def delete(recordId: String, recordType: Option[Type]): Unit =
    sqlUpdate(DELETE_DEPLOYMENT_RECORD_STMT, params("recordId" -> getName(recordId.normalized)), _ => ())

  override def update(facet: Facet): Facet = throw new UnsupportedOperationException("You cannot update records!")

  override def search(facetsFilters: FacetFilters): Seq[Facet] = searchRecords(facetsFilters, SELECT_ALL_DEPLOYMENT_RECORDS, deploymentRowMapper)

  override def exists(recordId: String, recordType: Option[Type]): Boolean = {
    recordType.foreach(requiresInstanceOfSupportedType)
    sqlQuery(EXISTS_DEPLOYMENT_RECORD_BY_ID, params("recordId" -> getName(recordId.normalized)), _.getInt(1) > 0).headOption.exists(identity)
  }

  override def findAllFacetsByRelease(release: Release): Seq[Facet] = findAllFacetsByReleaseId(release.getId).toSeq

  override def findAllFacetsByTask(task: Task): Seq[Facet] = findAllFacetsByTargetId(shortenId(task.getId)).toSeq

  override protected def findAllFacetsByReleaseId(releaseId: CiId): mutable.Seq[DeploymentRecord] =
    sqlQuery[DeploymentRecord](SELECT_DEPLOYMENT_RECORDS_BY_RELEASE_ID_STMT, params(
      "releaseId" -> FullReleaseId(releaseId).withOnlyOneParentOrApplicationsForArchiveDb()
    ), deploymentRowMapper)

  override protected def findAllFacetsByTargetId(targetId: CiId): mutable.Seq[DeploymentRecord] =
    sqlQuery(SELECT_DEPLOYMENT_RECORDS_BY_TASK_ID_STMT, params("taskId" -> targetId), deploymentRowMapper)

  def findAllApplicationNames: Set[String] =
    sqlQuery[String](FIND_ALL_APPLICATION_NAMES, params(), (rs: ResultSet, _: Int) => rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE.APPLICATION_NAME))
      .toSet

  def findAllEnvironmentNames: Set[String] =
    sqlQuery[String](FIND_ALL_ENVIRONMENT_NAMES, params(), (rs: ResultSet, _: Int) => rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE.ENVIRONMENT_NAME))
      .toSet

  private val deploymentRowMapper: RowMapper[DeploymentRecord] = (rs: ResultSet, _: Int) => {
    val deploymentRecord: DeploymentRecord = Type.valueOf(rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE.RECORD_TYPE))
      .getDescriptor.newInstance[DeploymentRecord](rs.getString(DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE.RECORD_ID))
    val taskId = {
      val id = rs.getString(REPORT_TASKS_ID_COLUMN)
      if (id.startsWith("Folder")) {
        APPLICATIONS.getRootNodeName + Ids.SEPARATOR + id
      } else id
    }
    deploymentItemRowMapper(rs, deploymentRecord, taskId, DEPLOYMENT_TASK_REPORTING_RECORD_ARCHIVE)
  }

}
