package com.xebialabs.xlrelease.reports.audit

import com.xebialabs.deployit.core.{ListOfStringView, MapStringStringView, SetOfStringView}
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, PropertyDescriptor}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.domain.CustomScriptTask.PYTHON_SCRIPT_PREFIX
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.planner.PlannerReleaseTree
import com.xebialabs.xlrelease.reports.audit.CommonFormat.{PreResolvedProperties, PropertyName, TaskAgent}
import com.xebialabs.xlrelease.reports.audit.TaskDetailsReport._
import com.xebialabs.xlrelease.reports.domain.MaybeData
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.excel._
import com.xebialabs.xlrelease.variable.VariableHelper._
import com.xebialabs.xlrelease.views.DependencyNode
import com.xebialabs.xlrelease.views.converters.DependencyViewConverter

import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

object TaskDetailsReport {
  val SECTION_LABEL: String = "Task details"
  val COMMENT_HEADER = ExcelHeaderColumn("Comment", 100)
  val INPUT_PROPERTY_HEADER = ExcelHeaderColumn("Input property", 45)
  val INPUT_VALUE_HEADER = ExcelHeaderColumn("Input value", 45)
  val OUTPUT_PROPERTY_HEADER = ExcelHeaderColumn("Output property", 45)
  val OUTPUT_VALUE_HEADER = ExcelHeaderColumn("Output value", 45)

  val NUMBER_OF_COLUMNS = 5

  case class Data(releaseTree: PlannerReleaseTree, taskAgents: Map[String, TaskAgent])

  private case class TaskProperty(key: String, value: Maybe[Any])

  private case class CommentPropertyCells(comment: Option[Comment], inputProperty: Option[TaskProperty], outputProperty: Option[TaskProperty])

  def addSection(data: Data,
                 workbook: ReportWorkbook,
                 styles: ExcelStyles,
                 overviewSheetWriter: ExcelSheetWriter,
                 taskTitleCellsForHyperlinks: Map[String, (Int, Int)]
                ): Unit = {
    new TaskDetailsReport(data, workbook, styles, overviewSheetWriter).addContent(taskTitleCellsForHyperlinks)
  }

  type PropertyValueFormatter[A] = A => String
  type PerTaskTypePropertyFormatters = Seq[(String, PropertyValueFormatter[_])]

  // ensure uniqueness of property names, In case of duplicates, the first entry is kept.
  def PerTaskTypePropertyFormatters(entries: (String, PropertyValueFormatter[_])*): PerTaskTypePropertyFormatters =
    entries.foldLeft(Set.empty[String] -> Seq.empty[(String, PropertyValueFormatter[_])]) {
      case ((seen, result), (propertyName, _)) if seen contains propertyName => seen -> result
      case ((seen, result), entry@(propertyName, _)) => (seen + propertyName) -> (result :+ entry)
    }._2

  private val CREATE_RELEASE_TASK_INPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "templateVariables" -> formatVariables,
    "releaseTags" -> formatSetString,
    "startRelease" -> formatAsIs,
    "folderId" -> formatAsIs,
    "templateId" -> formatAsIs,
    "newReleaseTitle" -> formatAsIs
  )

  private val CREATE_RELEASE_TASK_OUTPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "createdReleaseId" -> formatAsIs
  )

  private val GATE_TASK_INPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "conditions" -> formatConditions,
    "dependencies" -> formatDependencies
  )

  private val NOTIFICATION_TASK_INPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "body" -> formatAsIs,
    "priority" -> formatAsIs,
    "subject" -> formatAsIs,
    "replyTo" -> formatAsIs,
    "bcc" -> formatSetString,
    "cc" -> formatSetString,
    "addresses" -> formatSetString
  )

  private val USER_INPUT_TASK_INPUT: PerTaskTypePropertyFormatters =
    Seq("variables" -> formatVariables)

  private val SCRIPT_TASK_INPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "script" -> formatAsIs
  )

  private val EXTERNAL_SCRIPT_TASK_INPUT: PerTaskTypePropertyFormatters = PerTaskTypePropertyFormatters(
    "url" -> formatAsIs,
    "username" -> formatAsIs,
    "password" -> formatAsIs,
    "scriptEngine" -> formatAsIs
  )

  private def formatVariables: PropertyValueFormatter[java.util.List[Variable]] =
    _.asScala.map(v => s"${v.getKey}=${v.getValue}").mkString("{", ", ", "}")

  private def formatAsIs: PropertyValueFormatter[Any] = a => if (a != null) a.toString else ""

  private def formatSetString: PropertyValueFormatter[java.util.Set[String]] = _.asScala.mkString("[", ", ", "]")

  private def formatConditions: PropertyValueFormatter[java.util.List[GateCondition]] =
    _.asScala.map(v => s"${v.getTitle}=${v.isChecked}").mkString("{", ", ", "}")

  private def formatDependencies: PropertyValueFormatter[java.util.List[Dependency]] =
    _.asScala.map(DependencyViewConverter.toDependencyView).map(view => {

      def getDependencyTitle(node: DependencyNode): String =
        Seq(Option(node.getReleaseTitle), Option(node.getPhaseTitle), Option(node.getTaskTitle)).flatten.mkString(" / ")

      if (view.getVariableOrTarget.getValue != null) {
        s"${getDependencyTitle(view.getVariableOrTarget.getValue)}=${view.isResolved}"
      } else if (view.getVariableOrTarget.getVariable != null) {
        s"${view.getVariableOrTarget.getVariable}=${view.isResolved}"
      } else {
        view.getId
      }
    }).mkString("{", ", ", "}")

  def getPropertyValue(release: Release,
                       item: BaseConfigurationItem,
                       pd: PropertyDescriptor,
                       variableMapping: mutable.Map[String, String],
                       replaceVariables: Boolean,
                       isScriptInterpolationOff: Boolean = false): Option[Any] = {

    val fqPropertyName = PYTHON_SCRIPT_PREFIX + pd.getName
    val value = if (variableMapping.contains(fqPropertyName)) {
      val variableName = variableMapping(fqPropertyName)
      Option(release.getVariablesByKeys.get(withoutVariableSyntax(variableName))).map(_.getValue).orNull
    } else {
      item.getProperty[AnyRef](pd.getName)
    }

    if (value == null) {
      None
    } else if (replaceVariables) {
      replaceVariableWithVariableValues(release, pd, value, isScriptInterpolationOff)
    } else {
      Some(value)
    }
  }

  def replaceVariableWithVariableValues(release: Release, pd: PropertyDescriptor, value: AnyRef, isScriptInterpolationOff: Boolean = false): Option[Any] = {
    val result = if (pd.getName == CustomScriptTask.IGNORE_SCRIPT_VARIABLE_INTERPOLATION && isScriptInterpolationOff) {
      value
    } else if (pd.isPassword) {
      replaceAll(value, release.getPasswordVariableValues, mutable.Set.empty[String].asJava, freezeEvenIfUnresolved = false)
    } else {
      replaceAllWithInterpolation(value, release.getAllVariableValuesAsStringsWithInterpolationInfo, mutable.Set.empty[String].asJava, freezeEvenIfUnresolved = false)
    }
    Option(result)
  }
}

class TaskDetailsReport(data: Data,
                        val workbook: ReportWorkbook,
                        val styles: ExcelStyles,
                        val overviewSheetWriter: ExcelSheetWriter) extends CommonTaskReportSheet with ErrorHandler {

  var sheetWriter: ExcelSheetWriter = workbook.createReportSection(SECTION_LABEL)

  val release: Release = data.releaseTree.release

  def addHeader(): Unit = {
    sheetWriter.newRow()
    addTaskCommonHeader()
    sheetWriter
      .addHeaderCell(INPUT_PROPERTY_HEADER, styles.whiteOnGreen)
      .addHeaderCell(INPUT_VALUE_HEADER, styles.whiteOnGreen)
      .addHeaderCell(OUTPUT_PROPERTY_HEADER, styles.whiteOnGreen)
      .addHeaderCell(OUTPUT_VALUE_HEADER, styles.whiteOnGreen)
      .addHeaderCell(COMMENT_HEADER, styles.whiteOnGreen)
  }

  def addTaskSeparatorBorder(): Unit = {
    (0 until CommonTaskReportSheet.COLUMNS.length + TaskDetailsReport.NUMBER_OF_COLUMNS).map { idx =>
      Option(sheetWriter.getSheet.getRow(sheetWriter.getRowIndex - 1).getCell(idx)).getOrElse {
        sheetWriter.getSheet.getRow(sheetWriter.getRowIndex - 1).createCell(idx)
      }
    }.foreach { cell =>
      sheetWriter.addBottomBorderToCell(cell)
    }
  }

  private def replaceTaskTitleWithHyperlink(taskTitleCellsForHyperlinks: Map[String, (Int, Int)], task: Task, targetRow: Int, targetCol: Int): Unit = {
    taskTitleCellsForHyperlinks.get(task.getId).foreach {
      case (row, column) => overviewSheetWriter.replaceWithHyperlink(row, column, SECTION_LABEL, targetRow, targetCol, styles.hyperlink)
    }
  }

  private def hasDateInside(value: Maybe[Any]): Boolean = {
    value.hasType(classOf[Date])
  }

  def addContent(taskTitleCellsForHyperlinks: Map[String, (Int, Int)]): Unit = {
    addHeader()
    release.getPhases.asScala.foreach { phase =>
      phase.getAllTasks.asScala.foreach { task =>
        val taskAgent = data.taskAgents.getOrElse(task.getId, TaskAgent.empty)
        val commentPropertyCells: Seq[CommentPropertyCells] = getCommentPropertyCells(task, taskAgent.preResolvedProperties)
        if (commentPropertyCells.isEmpty) {
          sheetWriter.newRow()
          replaceTaskTitleWithHyperlink(taskTitleCellsForHyperlinks, task, sheetWriter.getRowIndex - 1, sheetWriter.getColumnIndex + 1)
          addTaskCommonCells(task, release, taskAgent)
        }
        commentPropertyCells.zipWithIndex.foreach {
          case (it, index) =>
            sheetWriter.newRow()
            if (index == 0) {
              replaceTaskTitleWithHyperlink(taskTitleCellsForHyperlinks, task, sheetWriter.getRowIndex - 1, sheetWriter.getColumnIndex + 1)
            }
            addTaskCommonCells(task, release, taskAgent)
            addCell(it.inputProperty.map(_.key).getOrElse(""))
            val inputPropertyValue = it.inputProperty.flatMap(property => Option(property.value)).getOrElse(MaybeData.success(""))
            truncateAndAddCellForMaybe(inputPropertyValue, if (hasDateInside(inputPropertyValue)) {
              styles.wrappedDate
            } else {
              styles.wrapped
            })
            addCell(it.outputProperty.map(_.key).getOrElse(""))
            val outputPropertyValue = it.outputProperty.flatMap(property => Option(property.value)).getOrElse(MaybeData.success(""))
            truncateAndAddCellForMaybe(outputPropertyValue, if (hasDateInside(outputPropertyValue)) {
              styles.wrappedDate
            } else {
              styles.wrapped
            })
            addCellForMaybe(MaybeData(formatCommentCell(it.comment)), styles.wrapped)
        }
        addTaskSeparatorBorder()
      }
    }
  }

  private def addCell(value: Any): Unit = {
    value match {
      case date: Date => sheetWriter.addCell(date)
      case int: Integer => sheetWriter.addCell(int, styles.leftAligned)
      case _ => sheetWriter.addCell(value)
    }
  }

  private def formatCommentCell(comment: Option[Comment]): String = {
    comment.map(CommonFormat.getComment).getOrElse("")
  }

  private def getCommentPropertyCells(task: Task, preResolvedProperties: PreResolvedProperties): Seq[CommentPropertyCells] = {
    val comments: Seq[Option[Comment]] = task.getComments.asScala.map(Option.apply).toSeq// comments are ordered from oldest to newer
    val inputProps: Seq[Option[TaskProperty]] = getInputProperties(task, preResolvedProperties).map(input => Option(TaskProperty.tupled(input))).toSeq
    val outputProps: Seq[Option[TaskProperty]] = getOutputProperties(task, preResolvedProperties).map(output => Option(TaskProperty.tupled(output))).toSeq
    val commentPropertyCells = comments
      .zipAll(inputProps, Option.empty[Comment], Option.empty[TaskProperty])
      .zipAll(outputProps, (Option.empty, Option.empty), Option.empty[TaskProperty])
      .map({
        case ((c, i), o) => CommentPropertyCells(c, i, o)
      })
    commentPropertyCells
  }

  private def getOutputProperties(task: Task, preResolvedProperties: PreResolvedProperties): Iterable[(String, Maybe[Any])] = {
    task match {
      case task: CustomScriptTask =>
        getProperties(task.getPythonScript, task.getPythonScript.getOutputProperties.asScala, preResolvedProperties,
          task.getVariableMapping.asScala, Function.const(false))
      case task: CreateReleaseTask =>
        getCoreProperties(task, preResolvedProperties, CREATE_RELEASE_TASK_OUTPUT)
      case _ => Seq.empty
    }
  }

  private def getInputProperties(task: Task, preResolvedProperties: PreResolvedProperties): Iterable[(String, Maybe[Any])] = {
    task match {
      case task: CustomScriptTask =>
        val isScriptInterpolationOff = task.isPropertyVariableInterpolationOff(task.getPythonScript)
        val replaceVariables = (pd: PropertyDescriptor) => task.getPythonScript.getPropertiesWithVariables.contains(pd)
        getProperties(task.getPythonScript, task.getPythonScript.getInputProperties.asScala, preResolvedProperties,
          task.getVariableMapping.asScala, replaceVariables, isScriptInterpolationOff)
      case task: CreateReleaseTask =>
        getCoreProperties(task, preResolvedProperties, CREATE_RELEASE_TASK_INPUT)
      case task: GateTask =>
        getCoreProperties(task, preResolvedProperties, GATE_TASK_INPUT)
      case task: NotificationTask =>
        getCoreProperties(task, preResolvedProperties, NOTIFICATION_TASK_INPUT)
      case task: UserInputTask =>
        getCoreProperties(task, preResolvedProperties, USER_INPUT_TASK_INPUT)
      case task: ScriptTask =>
        getCoreProperties(task, preResolvedProperties, SCRIPT_TASK_INPUT)
      case task: ResolvableScriptTask =>
        getCoreProperties(task, preResolvedProperties, EXTERNAL_SCRIPT_TASK_INPUT)
      case _ => Seq.empty
    }
  }

  private def propertyLabelAndFormattedValue(item: BaseConfigurationItem,
                                             itemDescriptor: Descriptor,
                                             preResolvedProperties: PreResolvedProperties)
                                            (propertyName: PropertyName, formatter: PropertyValueFormatter[_]): Option[(String, Maybe[String])] = {
    preResolvedProperties.get(propertyName)
      .orElse(Option(MaybeData.success(formatter(item.getProperty(propertyName)))))
      .map(itemDescriptor.getPropertyDescriptor(propertyName).getLabel -> _)
  }

  private def getCoreProperties(item: BaseConfigurationItem,
                                preResolvedProperties: PreResolvedProperties,
                                properties: PerTaskTypePropertyFormatters): Iterable[(String, Maybe[String])] = {
    val getLabelAndValue: (PropertyName, PropertyValueFormatter[_]) => Option[(String, Maybe[String])] =
      propertyLabelAndFormattedValue(item, item.getType.getDescriptor, preResolvedProperties)

    properties.flatMap(getLabelAndValue.tupled(_))
  }

  private def getProperties(item: BaseConfigurationItem,
                            properties: Iterable[PropertyDescriptor],
                            preResolvedProperties: PreResolvedProperties,
                            variableMapping: mutable.Map[String, String],
                            replaceVariables: PropertyDescriptor => Boolean,
                            isScriptInterpolationOff: Boolean = false): Iterable[(String, Maybe[Any])] = {
    properties.map { pd =>
      pd.getLabel -> {
        preResolvedProperties.getOrElse(pd.getName,
          getPropertyValue(release, item, pd, variableMapping, replaceVariables(pd))
            .map(showPropertyValue)
            // if property value is null - it is still valid value
            .getOrElse(MaybeData.success(null))
        )
      }
    }
  }

  private def showPropertyValue(value: Any): Maybe[Any] = {
    MaybeData.success(value match {
      case map: MapStringStringView => map.asScala.map {
        case (key, v) => s"$key=$v"
      }.mkString("{", ", ", "}")
      case set: SetOfStringView => set.asScala.mkString("[", ", ", "]")
      case list: ListOfStringView => list.asScala.mkString("[", ", ", "]")
      case conf: Configuration => conf.getTitle
      case other => other
    })
  }
}
