package com.xebialabs.xlrelease.reports.job.impl

import com.codahale.metrics.InstrumentedExecutorService
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.metrics.XlrMetricRegistry
import com.xebialabs.xlrelease.reports.job.impl.ReportExecutorService._
import com.xebialabs.xlrelease.utils.PrefixedThreadFactory
import grizzled.slf4j.Logging

import java.util.concurrent.{Callable, Executors, Future, TimeUnit}
import scala.concurrent.{ExecutionContext, ExecutionContextExecutorService}

class ReportExecutorService(xlrConfig: XlrConfig) extends Logging {

  private val executorService = {
    val executor = Executors.newFixedThreadPool(xlrConfig.reporting.engine.maxThreadsCount, new PrefixedThreadFactory(REPORT_POOL_NAME, true))
    if (xlrConfig.metrics.enabled) {
      new InstrumentedExecutorService(executor, XlrMetricRegistry.metricRegistry, REPORT_POOL_NAME)
    } else {
      executor
    }
  }

  lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(executorService)

  def shutdown(): Unit = {
    logger.info(s"Shutting down thread pool '$REPORT_POOL_NAME'")
    executorService.shutdown()
    try {
      val timeout = 60
      if (!executorService.awaitTermination(timeout, TimeUnit.SECONDS)) {
        executorService.shutdownNow()
        if (!executorService.awaitTermination(timeout, TimeUnit.SECONDS)) {
          logger.error(s"Unable to shutdown thread pool '$REPORT_POOL_NAME' in ${2 * timeout} seconds")
        }
      }
      logger.info(s"Shut down thread pool '$REPORT_POOL_NAME'")
    } catch {
      case _: InterruptedException =>
        logger.info(s"Interrupted while shutting down thread pool '$REPORT_POOL_NAME'")
        executorService.shutdownNow()
        Thread.currentThread().interrupt()
    }
  }

  def submit[T](task: Callable[T]): Future[T] = executorService.submit(task)
}

object ReportExecutorService {
  final val REPORT_POOL_NAME = "report"
}
