package com.xebialabs.xlrelease.reports.db

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.db.sql.archiving.SelectArchivedTasksBuilder
import com.xebialabs.xlrelease.domain.CustomScriptTask.UNKNOWN_TYPE
import com.xebialabs.xlrelease.domain.TaskGroup
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus.{COMPLETED, COMPLETED_IN_ADVANCE}
import com.xebialabs.xlrelease.reports.dto._
import com.xebialabs.xlrelease.reports.service.{ReportParams, ReportService}
import com.xebialabs.xlrelease.views.converters.UserViewConverter
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Repository
class ArchivedTasksReports @Autowired()(@Qualifier("reportingJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                        @Qualifier("reportingSqlDialect") implicit val reportingSqlDialect: Dialect,
                                        implicit val permissionEnforcer: PermissionEnforcer,
                                        implicit val roleService: RoleService,
                                        viewConverter: UserViewConverter) extends Logging {

  private lazy val excludedTaskTypes: Set[Type] = DescriptorRegistry.getSubtypes(Type.valueOf(classOf[TaskGroup])).asScala.toSet + UNKNOWN_TYPE

  def getTopLongestTasks(reportParams: ReportParams): JList[TaskDuration] = {

    val sqlQuery: SelectArchivedTasksBuilder = new SelectArchivedTasksBuilder(s"$REPORT_TASKS_TABLE_ALIAS.*")
      .withOneOfStatuses(COMPLETED.value(), COMPLETED_IN_ADVANCE.value())
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withNotNull(REPORT_TASKS_DURATION_COLUMN)
      .excludingTypes(excludedTaskTypes)
      .withFilters(reportParams.filters)
      .orderBy(s"$REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN DESC")
      .innerJoin(REPORT_RELEASES_TABLE_NAME, REPORT_RELEASES_TABLE_ALIAS, REPORT_TASKS_RELEASEID_COLUMN, COMMON_RELEASEID_COLUMN)
      .limit(ReportService.TOP_REPORT_SIZE)

    val (sql, params) = sqlQuery.build()
    jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) =>
      new TaskDuration(
        viewConverter.toUserView(rs.getString(REPORT_TASKS_OWNER_COLUMN)),
        rs.getString(REPORT_TASKS_TITLE_COLUMN),
        rs.getLong(REPORT_TASKS_DURATION_COLUMN),
        rs.getString(REPORT_TASKS_RELEASE_TITLE_COLUMN),
        rs.getString(REPORT_TASKS_PHASE_TITLE_COLUMN),
        rs.getString(REPORT_TASKS_TEAM_COLUMN),
        rs.getString(REPORT_TASKS_TASK_TYPE_COLUMN),
        rs.getString(REPORT_TASKS_RELEASEID_COLUMN)
      ))
  }

  def getTopLongestTaskTypes(reportParams: ReportParams): JList[LongestTaskType] = {
    val sqlQuery: SelectArchivedTasksBuilder = new SelectArchivedTasksBuilder(
      s"""
         |$REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_TASK_TYPE_COLUMN,
         |AVG($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) AS averageDuration,
         |COUNT(DISTINCT $REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_RELEASEID_COLUMN) AS containedInNumberOfReleases""".stripMargin)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withNotNull(REPORT_TASKS_TASK_TYPE_COLUMN)
      .excludingTypes(excludedTaskTypes)
      .withFilters(reportParams.filters)
      .innerJoin(REPORT_RELEASES_TABLE_NAME, REPORT_RELEASES_TABLE_ALIAS, REPORT_TASKS_RELEASEID_COLUMN, COMMON_RELEASEID_COLUMN)
      .groupBy(s"$REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_TASK_TYPE_COLUMN")
      .orderBy(s"AVG($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) DESC")
      .limit(ReportService.TOP_REPORT_SIZE)
    val (sql, params) = sqlQuery.build()
    jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) =>
      new LongestTaskType(
        rs.getString(REPORT_TASKS_TASK_TYPE_COLUMN),
        rs.getLong("averageDuration"),
        rs.getLong("containedInNumberOfReleases")
      ))
  }

  def getAverageAndLongestTaskDuration(reportParams: ReportParams): AverageAndLongestTaskDuration = {
    val sqlQuery: SelectArchivedTasksBuilder = new SelectArchivedTasksBuilder(
      s"""
         |AVG($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) AS averageTaskDuration,
         |MAX($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) AS maxTaskDuration
         |""".stripMargin)
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .excludingTypes(excludedTaskTypes)
      .withFilters(reportParams.filters)
      .innerJoin(REPORT_RELEASES_TABLE_NAME, REPORT_RELEASES_TABLE_ALIAS, REPORT_TASKS_RELEASEID_COLUMN, COMMON_RELEASEID_COLUMN)
    val (sql, params) = sqlQuery.build()
    jdbcTemplate.queryForObject(sql, params.toArray, (rs: ResultSet, _: Int) =>
      new AverageAndLongestTaskDuration(
        rs.getLong("averageTaskDuration"),
        rs.getLong("maxTaskDuration")
      ))
  }

  def getReleaseEfficiency(reportParams: ReportParams): ReleaseEfficiency = {
    val taskCounter = countTasks(reportParams) _

    val (automatedTasksCount, automatedTasksDuration) = taskCounter(true)
    val (manualTasksCount, manualTasksDuration) = taskCounter(false)

    val releaseEfficiency = new ReleaseEfficiency()
    releaseEfficiency.setTotalAutomatedTasks(automatedTasksCount)
    releaseEfficiency.setTotalManualTasks(manualTasksCount)
    releaseEfficiency.setTotalTasks(automatedTasksCount + manualTasksCount)
    releaseEfficiency.setTotalAutomatedTimeSpent(automatedTasksDuration)
    releaseEfficiency.setTotalManualTimeSpent(manualTasksDuration)
    releaseEfficiency.setTotalTimeSpent(automatedTasksDuration + manualTasksDuration)
    releaseEfficiency
  }

  private def countTasks(reportParams: ReportParams)(automated: Boolean): (Long, Long) = {
    val query = new SelectArchivedTasksBuilder(
      s"COUNT($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_ID_COLUMN) as tasksCount",
      s"SUM($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) as tasksDuration")
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withAutomated(automated)
      .withReleaseStatus(ReleaseStatus.COMPLETED)
      .excludingTypes(excludedTaskTypes)
      .withFilters(reportParams.filters)
      .innerJoin(REPORT_RELEASES_TABLE_NAME, REPORT_RELEASES_TABLE_ALIAS, REPORT_TASKS_RELEASEID_COLUMN, COMMON_RELEASEID_COLUMN)


    val (sql, params) = query.build()
    jdbcTemplate.queryForObject(sql, params.toArray, (rs: ResultSet, _: Int) =>
      (rs.getLong("tasksCount"), rs.getLong("tasksDuration")))
  }

  def getTopPeopleInvolved(reportParams: ReportParams): JList[UserParticipation] = {
    val query = new SelectArchivedTasksBuilder(
      s"COUNT($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_ID_COLUMN) as tasksCount",
      s"SUM($REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_DURATION_COLUMN) as tasksDuration",
      s"$REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_OWNER_COLUMN")
      .withOneOfStatuses(COMPLETED.value(), COMPLETED_IN_ADVANCE.value())
      .withDates(reportParams.timeFrame, reportParams.from, reportParams.to)
      .withTags(reportParams.tags)
      .withSecurity(reportParams.userSpecific)
      .withFilters(reportParams.filters)
      .withNotNull(REPORT_TASKS_OWNER_COLUMN)
      .innerJoin(REPORT_RELEASES_TABLE_NAME, REPORT_RELEASES_TABLE_ALIAS, REPORT_TASKS_RELEASEID_COLUMN, COMMON_RELEASEID_COLUMN)
      .groupBy(s"$REPORT_TASKS_TABLE_ALIAS.$REPORT_TASKS_OWNER_COLUMN")
      .orderBy("tasksDuration DESC")
      .limit(ReportService.TOP_REPORT_SIZE)

    val (sql, params) = query.build()
    jdbcTemplate.query(sql, params.toArray, (rs: ResultSet, _: Int) => {
      val tasksCount = rs.getInt("tasksCount")
      val tasksDuration = rs.getLong("tasksDuration")
      val taskOwner = rs.getString(REPORT_TASKS_OWNER_COLUMN)

      new UserParticipation(viewConverter.toUserView(taskOwner), tasksDuration, tasksCount)
    })
  }
}
