package com.xebialabs.xlrelease.reports.audit

import com.google.common.base.Strings
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, Release, Trigger}
import com.xebialabs.xlrelease.reports.domain._
import com.xebialabs.xlrelease.reports.service.ReportsService
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository, TriggerRepository}
import com.xebialabs.xlrelease.service.ReleaseActivityLogsService

import scala.jdk.CollectionConverters._
import scala.util.Try

object CreatedByDetailsHelper {
  def getCreationDetails(releaseRepository: ReleaseRepository,
                         triggerRepository: TriggerRepository,
                         activityLogsService: ReleaseActivityLogsService,
                         resolveUser: String => Option[String])
                        (releaseData: ReleaseData): Option[ReleaseStartedBy] = {
    val release = releaseData.release
    if (release.isCreatedFromTrigger) {
      Some(tryToFetchTriggerInfo(triggerRepository)(release))
    } else if (!Strings.isNullOrEmpty(release.getStartedFromTaskId)) {
      val masterReleaseId = Ids.releaseIdFrom(release.getStartedFromTaskId)
      Some(StartedByCreateReleaseTask(Try {
        releaseRepository.getTitle(masterReleaseId)
      }, masterReleaseId))
    } else {
      val startedBy = if (release.isTemplate || release.isPlanned) {
        None
      } else {
        releaseData.startedByLogEntry
          .map(getUserFromActivityLogEntry(resolveUser))
      }
      val createdBy = releaseData.createdByLogEntry
        .map(getUserFromActivityLogEntry(resolveUser))
      Some(StartedByUser(
        startedBy,
        createdBy)
      )
    }
  }

  def tryToFetchTriggerInfo(triggerRepository: TriggerRepository)
                           (release: Release): StartedByTrigger = {
    release.getTags.asScala.find(e => e.startsWith("Trigger")).map {
      triggerTag =>
        StartedByTrigger(Try(triggerRepository.find[Trigger](triggerTag)), Option(triggerTag))
    } match {
      case Some(startedByTrigger) => startedByTrigger
      case _ => StartedByTrigger(Try(throw new NotFoundException(s"Trigger tag is missing")), Option.empty)
    }
  }

  def getUserFromActivityLogEntry(resolveUser: String => Option[String])
                                 (activityLogEntry: ActivityLogEntry): UserThatCreatedOrStartedRelease = {
    val username = Option(activityLogEntry.getUsername).getOrElse(ReportsService.SYSTEM_USER_NAME)
    UserThatCreatedOrStartedRelease(
      username,
      Try {
        resolveUser(username)
      }.toOption.flatten
    )
  }
}
