package com.xebialabs.xlrelease.reports.excel;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.xebialabs.xlrelease.reports.domain.ReportCellError;
import com.xebialabs.xlrelease.reports.domain.ReportSheetError;

public class ReportWorkbook {

    private static final String WARNING_EMOJI_UNICODE = "\u26A0";
    private XSSFWorkbook workbook;

    private List<ReportCellError> cellErrors = new ArrayList<>();
    private List<ReportSheetError> sheetErrors = new ArrayList<>();
    private AtomicBoolean errorSheetAdded = new AtomicBoolean(false);

    public ReportWorkbook() {
        workbook = new XSSFWorkbook();
    }

    public XSSFWorkbook getWorkbookWithoutErrorSheet() {
        Export.preventDynamicDataExchangeAttack(workbook);
        return workbook;
    }

    public ExcelSheetWriter createReportSection(String title) {
        return new ExcelSheetWriter(this, workbook.createSheet(title));
    }

    public XSSFWorkbook makeWorkbookWithErrors(ExcelStyles styles) {
        addErrorSheetIfNecessary(styles);
        Export.preventDynamicDataExchangeAttack(workbook);
        return workbook;
    }

    private void addErrorSheetIfNecessary(ExcelStyles styles) {
        if (errorSheetAdded.getAndSet(true)) {
            throw new IllegalStateException("Error sheet added more than once");
        }
        if (!cellErrors.isEmpty() || !sheetErrors.isEmpty()) {
            ExcelSheetWriter reportSection = createReportSection(String.format("%s Report errors (%d)", WARNING_EMOJI_UNICODE, cellErrors.size()+sheetErrors.size()));
            new ErrorSheet(reportSection).addContent(styles);
        }
    }

    public void addCellError(ReportCellError error) {
        if (errorSheetAdded.get()) {
            throw new IllegalStateException("Error sheet already added, too late to submit new errors");
        }
        cellErrors.add(error);
    }

    public void addSheetError(ReportSheetError error) {
        if (errorSheetAdded.get()) {
            throw new IllegalStateException("Error sheet already added, too late to submit new errors");
        }
        sheetErrors.add(error);
    }

    public List<ReportCellError> getCellErrors() {
        return cellErrors;
    }

    public List<ReportSheetError> getSheetErrors() {
        return sheetErrors;
    }
}
