package com.xebialabs.xlrelease.reports.job.impl

import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.reports.job.api._
import com.xebialabs.xlrelease.reports.job.domain.{ReportJob, ReportJobStatus}
import com.xebialabs.xlrelease.reports.job.events.internal.{FinishedRunReportJob, ReportJobInstanceEnqueued}
import com.xebialabs.xlrelease.reports.job.events.{ReportJobAbortedEvent, ReportJobCompletedEvent, ReportJobFailedEvent}
import com.xebialabs.xlrelease.reports.job.impl.ReportJobInstance._
import com.xebialabs.xlrelease.reports.job.repository.ReportJobRepository
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationContext

import java.lang.Thread.currentThread
import java.net.URI
import java.time.Instant
import java.util.Date
import java.util.concurrent.Future

class ReportJobInstance(reportJob: ReportJob,
                        applicationContext: ApplicationContext,
                        reportingEngineConfiguration: ReportingEngineConfiguration,
                        reportExecutorService: ReportExecutorService,
                        reportJobRepository: ReportJobRepository,
                        globalReportStorage: GlobalReportStorage,
                        eventBus: XLReleaseEventBus
                       ) extends Logging {

  private var runningPromise: Option[Future[Option[ReportResult]]] = None

  @volatile
  var cancelled = false

  def getJobId: Int = reportJob.getJobId()

  def run(): Option[ReportResult] = {
    try {
      start()
      Some(complete(doRun()))
    } catch {
      case e: Throwable =>
        fail(e)
        None
    } finally {
      eventBus.publish(FinishedRunReportJob(getJobId))
    }
  }

  def enqueue(): Future[Option[ReportResult]] = {
    logger.debug(s"Enqueued ${reportJob.getJobId()}")
    eventBus.publish(ReportJobInstanceEnqueued(this))
    runningPromise = Option(reportExecutorService.submit(() => this.run()))
    runningPromise.get
  }

  def fail(exception: Throwable): Unit = {
    if (reportJob.getStatus == ReportJobStatus.ABORTED) {
      logger.warn(exception.getMessage)
    } else {
      logger.error(s"Error while running report job [${getJobId}]", exception)
    }
    if (reportJob.getStatus == ReportJobStatus.SUBMITTED || reportJob.getStatus == ReportJobStatus.STARTED) {
      reportJob.setEndTime(currentTime)
      reportJob.setStatus(ReportJobStatus.FAILED)
      reportJobRepository.update(reportJob)
      runningPromise = None
      eventBus.publish(ReportJobFailedEvent(reportJob))
    }
  }

  private def doRun(): URI = {
    val reportDefinition: ReportDefinition = reportJob.getReportDefinition
    logger.info(s"Starting report job [${reportDefinition.reportName}] from user [${reportDefinition.getGeneratedBy()}]")
    applicationContext.getAutowireCapableBeanFactory.autowireBean(reportDefinition)
    val jobId = reportJob.getJobId
    val reportStorage: ReportStorage = globalReportStorage.reportStorage(jobId, reportDefinition.getGeneratedOn())
    val reportJobProgressMonitor = new DefaultReportJobProgressMonitor(this)
    val runContext = ReportJobRunContext(jobId, reportJobProgressMonitor, reportStorage, reportJob.reportName)
    val resultFile = reportDefinition.run(runContext)
    val contextUri = reportStorage.workdir.toURI
    val resultUri = contextUri.relativize(resultFile.toURI)
    logger.info(s"Finished report job [${reportDefinition.reportName}] on [$resultUri]")
    resultUri
  }

  private def sendTotalWorkItems(totalItems: Int): Unit = {
    reportJob.setTotalWorkItems(totalItems)
    reportJobRepository.updateProgress(reportJob)
  }

  private def sendCompletedWorkItems(completedItems: Int): Unit = {
    // Sending completed work items for a job that is already marked as failed should have no effect.
    // For example, akka cluster will detect that node that runs job is removed from a cluster,
    // but thread that actually executes a job is still running and will update progress of a job.
    reportJob.setCompletedWorkItems(completedItems)
    reportJobRepository.updateProgress(reportJob)
  }

  def abort(): Unit = {
    logger.debug(s"Abort report job [${getJobId}]")
    if (reportJob.getStatus == ReportJobStatus.SUBMITTED || reportJob.getStatus == ReportJobStatus.STARTED) {
      cancelled = true

      reportJob.setEndTime(currentTime)
      reportJob.setStatus(ReportJobStatus.ABORTED)
      reportJobRepository.update(reportJob)
      eventBus.publish(ReportJobAbortedEvent(reportJob))

      runningPromise match {
        case Some(future) =>
          future.cancel(true)
          runningPromise = None
        case None =>
          logger.warn(s"RunningPromise is none for report job [${getJobId}]")
      }
    }
  }

  private def complete(resultUri: URI): ReportResult = {
    reportJob.setResultUri(resultUri.toString)
    reportJob.setEndTime(currentTime)
    reportJob.setStatus(ReportJobStatus.COMPLETED)
    reportJobRepository.update(reportJob)
    runningPromise = None
    eventBus.publish(ReportJobCompletedEvent(reportJob))
    globalReportStorage.reportResult(reportJob)
  }

  private def start(): Unit = {
    reportJob.setNode(reportingEngineConfiguration.node)
    reportJob.setStartTime(currentTime)
    reportJob.setStatus(ReportJobStatus.STARTED)
    reportJobRepository.update(reportJob)
  }

}

private[impl] object ReportJobInstance {
  private def currentTime: Date = Date.from(Instant.now())

  private class DefaultReportJobProgressMonitor(reportJobInstance: ReportJobInstance) extends ReportJobProgressMonitor {

    override def sendTotalWorkItems(totalItems: Int): Unit = {
      reportJobInstance.sendTotalWorkItems(totalItems)
    }

    override def sendCompletedWorkItems(completedItems: Int): Unit = {
      reportJobInstance.sendCompletedWorkItems(completedItems)
      if (reportJobInstance.cancelled || currentThread().isInterrupted) {
        throw new InterruptedException(s"Report job [${reportJobInstance.getJobId}] was aborted")
      }
    }
  }

}
