package com.xebialabs.xlrelease.reports.api.v1.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.function.Predicate;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.ReportApi;
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord;
import com.xebialabs.xlrelease.reports.job.domain.BaseStreamingReleaseReportDefinition;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.FacetService;
import com.xebialabs.xlrelease.service.ReleaseService;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.utils.TypeHelper.getAllSubtypesOf;
import static java.util.stream.Collectors.toList;
import static scala.jdk.javaapi.CollectionConverters.asJava;

@Controller
public class ReportApiImpl implements ReportApi {

    private final ApplicationContext applicationContext;
    private final FacetService facetService;
    private final ReleaseService releaseService;
    private final PermissionChecker permissions;

    @Autowired
    public ReportApiImpl(ApplicationContext applicationContext, FacetService facetService, ReleaseService releaseService, PermissionChecker permissions) {
        this.applicationContext = applicationContext;
        this.facetService = facetService;
        this.releaseService = releaseService;
        this.permissions = permissions;
    }

    @Timed
    @Override
    public List<TaskReportingRecord> getRecordsForRelease(String releaseId) {
        return searchRecords(new FacetFilters().withParentId(releaseId));
    }

    @Timed
    @Override
    public List<TaskReportingRecord> getRecordsForTask(String taskId) {
        return searchRecords(new FacetFilters().withTargetId(taskId));
    }

    @Timed
    @Override
    public List<TaskReportingRecord> searchRecords(FacetFilters facetFilters) {
        facetFilters.validate();
        String containerId = facetFilters.getParentId() != null ? facetFilters.getParentId() : facetFilters.getTargetId();
        permissions.checkView(releaseIdFrom(containerId));
        checkFiltersContainValidSearchableTypes(facetFilters);
        return facetService.search(facetFilters)
                .stream()
                .map(facet -> (TaskReportingRecord) facet)
                .collect(toList());
    }

    @Timed
    @Override
    public Response downloadReleaseReport(String reportType, String releaseId) {
        BaseStreamingReleaseReportDefinition report = createReportDefinition(reportType, releaseId);
        StreamingOutput output = report.run();
        return ResponseHelper.streamFile(report.getFileName(), output, report.contentType());
    }

    @Timed
    @Override
    public byte[] getReleaseReport(String reportType, String releaseId) throws IOException {
        BaseStreamingReleaseReportDefinition report = createReportDefinition(reportType, releaseId);
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            report.run().write(outputStream);
        } finally {
            outputStream.close();
        }
        return outputStream.toByteArray();
    }

    private BaseStreamingReleaseReportDefinition createReportDefinition(String reportType, String releaseId) {
        Release release = releaseService.findByIdIncludingArchived(releaseId);
        permissions.checkView(release);
        BaseStreamingReleaseReportDefinition report = Type.valueOf(reportType).getDescriptor().newInstance(null);
        applicationContext.getAutowireCapableBeanFactory().autowireBean(report);
        report.setRelease(release);
        return report;
    }

    private void checkFiltersContainValidSearchableTypes(FacetFilters facetFilters) {
        Type taskReportingRecordType = Type.valueOf(TaskReportingRecord.class);
        if (facetFilters.getTypes() == null || facetFilters.getTypes().isEmpty()) {
            facetFilters.setTypes(asJava(getAllSubtypesOf(taskReportingRecordType)));
        } else {
            Predicate<Type> isUnsupportedType = facetType -> !facetType.instanceOf(taskReportingRecordType);
            facetFilters.getTypes().removeIf(isUnsupportedType);
            if (facetFilters.getTypes().isEmpty()) {
                throw new IllegalArgumentException("Filter contains unsupported types");
            }
        }
    }
}
