package com.xebialabs.xlrelease.reports.audit

// scalastyle:off illegal.imports

import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.planner.{Planner, PlannerReleaseItem, PlannerReleaseTree}
import com.xebialabs.xlrelease.reports.audit.CommonFormat.TaskAgent
import com.xebialabs.xlrelease.reports.audit.ReleaseOverviewReport._
import com.xebialabs.xlrelease.reports.domain.MaybeData._
import com.xebialabs.xlrelease.reports.domain._
import com.xebialabs.xlrelease.reports.excel.AuditReport.{LOGO_HEIGHT, LOGO_WIDTH}
import com.xebialabs.xlrelease.reports.excel._
import org.apache.poi.ss.util.CellRangeAddress
import org.apache.poi.xssf.usermodel.XSSFCellStyle
import org.joda.time.DateTime

import java.awt.Color
import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.Success

object ReleaseOverviewReport {
  val SECTION_LABEL: String = "Release Overview"

  val TEMPLATE_TAGS = "Template Tags"
  val RELEASE_TAGS = "Release Tags"
  val WORKFLOW_TAGS = "Workflow Tags"
  val TEMPLATE_TITLE = "Template title"
  val RELEASE_TITLE = "Release title"
  val WORKFLOW_TITLE = "Workflow title"
  val TEMPLATE_INFO = "Template info"
  val RELEASE_INFO = "Release info"
  val TEMPLATE_FOLDER = "Template folder"

  val PHASE_COLUMN = ExcelHeaderColumn("Phase", 20)
  val TASK_COLUMN = ExcelHeaderColumn("Task", 38)
  val TASK_DESCRIPTION_COLUMN = ExcelHeaderColumn("Task Description", 50)
  val TYPE_COLUMN = ExcelHeaderColumn("Type", 16)
  val STATUS_COLUMN = ExcelHeaderColumn("Status", 18)
  val START_DATE_COLUMN = ExcelHeaderColumn("Start date", CommonFormat.DATE_WIDTH)
  val END_DATE_COLUMN = ExcelHeaderColumn("End date", CommonFormat.DATE_WIDTH)
  val DURATION_COLUMN = ExcelHeaderColumn("Duration", 16)
  val ASSIGNED_USER_COLUMN = ExcelHeaderColumn("Assigned user", 22)
  val ASSIGNED_TEAM_COLUMN = ExcelHeaderColumn("Assigned team", 18)
  val COMPLETED_BY_COLUMN = ExcelHeaderColumn("Completed by", 24)

  val TASK_COLUMNS = Seq(
    PHASE_COLUMN,
    TASK_COLUMN,
    TASK_DESCRIPTION_COLUMN,
    TYPE_COLUMN,
    STATUS_COLUMN,
    START_DATE_COLUMN,
    END_DATE_COLUMN,
    DURATION_COLUMN,
    ASSIGNED_USER_COLUMN,
    ASSIGNED_TEAM_COLUMN,
    COMPLETED_BY_COLUMN
  )
  val NUMBER_OF_COLUMNS: Int = TASK_COLUMNS.size

  val CREATED_BY = "Created by"
  val STARTED_BY = "Started by"
  val FOLDER = "Folder"
  val TEMPLATE = "Template"

  case class Data(generatedBy: String,
                  generatedOn: Date,
                  masterReportName: String,
                  instanceData: InstanceData,
                  releaseTree: PlannerReleaseTree,
                  releaseFolder: Maybe[String],
                  releaseOwner: Option[String],
                  startedBy: Option[ReleaseStartedBy],
                  templateInfo: Maybe[Option[ReleaseTitleAndPath]],
                  tasksAgents: Map[String, TaskAgent])

}

class ReleaseOverviewReport(data: ReleaseOverviewReport.Data,
                            val workbook: ReportWorkbook,
                            val styles: ExcelStyles) extends ExcelCommonHeader with CommonTaskReportSheet with ErrorHandler {

  var sheetWriter: ExcelSheetWriter = workbook.createReportSection(ReleaseOverviewReport.SECTION_LABEL)

  private def isStarted: Boolean = data.releaseTree.release.getStatus != ReleaseStatus.PLANNED

  private def addReleaseInfoRowWithMaybe[T](label: String, value: Maybe[T], style: XSSFCellStyle = null,
                                            color: Color = null, validationMessage: String = null,
                                            firstRowTitle: String = null): Unit = {
    sheetWriter.newRow()
    sheetWriter.addCell(Option(firstRowTitle).filterNot(_.isEmpty).getOrElse(""), styles.noGridBoldText)
    sheetWriter.addCell(Option(label).filterNot(_.isEmpty).getOrElse(""), styles.leftAlignedNoGrid)
    if (style == null) {
      addCellForMaybe(value, value match {
        case Success(Right(_: Date)) | Success(Left((_, _: Date))) => styles.noBorderDate(color)
        case _ => styles.noBorder(color)
      })
    } else {
      addCellForMaybe(value, style)
    }
    sheetWriter.mergeCells(new CellRangeAddress(
      sheetWriter.getRowIndex - 1, sheetWriter.getRowIndex - 1,
      sheetWriter.getColumnIndex - 1, NUMBER_OF_COLUMNS - sheetWriter.getColumnIndex
    ))
    fillCellsUntilEndOfRow()
  }

  def addTaskHeader(): Unit = {
    TASK_COLUMNS.foreach(sheetWriter.addHeaderCell(_, styles.whiteOnGreen))
  }

  def addContent(): Map[String, (Int, Int)] = {
    val reportInfo = if (data.releaseTree.release.isTemplate) {
      ReportInfo(reportTitle = TEMPLATE_HEADER_TITLE, masterReportName = data.masterReportName)
    } else {
      ReportInfo(masterReportName = data.masterReportName)
    }

    addCommonHeader(data.generatedBy, data.generatedOn, data.instanceData,
      ReleaseOverviewReport.NUMBER_OF_COLUMNS,
      LOGO_WIDTH, LOGO_HEIGHT,
      reportInfo
    )

    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)

    Planner.makePlan(data.releaseTree, DateTime.now())
    addReleaseInformation()
    addTaskHeader()
    data.releaseTree.root.getPhases().asScala.flatMap(plannedPhase => {
      addPhaseRow(plannedPhase)
      plannedPhase.getAllTasks().asScala.flatMap {
        plannedTask => {
          addTaskRow(plannedTask).map(plannedTask.id -> _)
        }
      }
    }
    ).toMap
  }

  private def addReleaseInformation(): Unit = {
    val release = data.releaseTree.release

    addReleaseAndTemplateRows(release)
    addNoBorderEmptyRow(NUMBER_OF_COLUMNS)
    addOtherReleaseInfoRows(data)

    sheetWriter.newRow()
    fillCellsUntilEndOfRow()
    sheetWriter.newRow()
  }

  private def fillCellsUntilEndOfRow(): Unit = {
    while (sheetWriter.getColumnIndex < NUMBER_OF_COLUMNS) {
      sheetWriter.addCell("", styles.noBorder())
    }
  }

  private def addReleaseAndTemplateRows(release: Release): Unit = {
    val (title, info) = if (release.isTemplate) {
      (TEMPLATE_TITLE, TEMPLATE_INFO)
    } else if (release.isWorkflow) {
      (WORKFLOW_TITLE, RELEASE_INFO)
    } else {
      (RELEASE_TITLE, RELEASE_INFO)
    }

    addReleaseInfoRowWithMaybe(title,
      MaybeData(Option(CommonFormat.getPlanItemTitle(release, release)).filter(_.trim.nonEmpty).orNull), firstRowTitle = info)

    addReleaseInfoRowWithMaybe("Description", MaybeData.allowingNull(release.getDescription))

    addReleaseInfoRowWithMaybe(FOLDER, data.releaseFolder)

    if (!release.isTemplate) {
      addReleaseInfoRowWithMaybe(TEMPLATE, data.templateInfo.flatMapMaybe {
        case Some(releaseTitleAndPath) => MaybeData(releaseTitleAndPath.title)
        case _ => MaybeData("")
      })
      addReleaseInfoRowWithMaybe(TEMPLATE_FOLDER, data.templateInfo.flatMapMaybe {
        case Some(releaseTitleAndPath) => releaseTitleAndPath.path
        case _ => MaybeData("")
      })
    }

  }

  //noinspection ScalaStyle
  private def addOtherReleaseInfoRows(data: Data): Unit = {
    def release = data.releaseTree.release

    if (!release.isTemplate) {
      addReleaseInfoRowWithMaybe("Status",
        MaybeData(Option(release.getStatus).map(s => CommonFormat.prettifyStatusString(s.value)).orNull))

      if (release.isWorkflow) {
        addReleaseInfoRowWithMaybe("Workflow owner", MaybeData(data.releaseOwner.filter(_.trim.nonEmpty).orNull))
      } else {
        addReleaseInfoRowWithMaybe("Release owner", MaybeData(data.releaseOwner.filter(_.trim.nonEmpty).orNull))
      }

      data.startedBy.foreach {
        case StartedByUser(startedBy, createdBy) =>
          addReleaseInfoRowWithMaybe(
            CREATED_BY,
            CommonFormat.formatCreatedByUser(createdBy))
          if (isStarted) {
            addReleaseInfoRowWithMaybe(
              STARTED_BY,
              CommonFormat.formatStartedByUser(startedBy))
          }
        case startedByTrigger: StartedByTrigger =>
          addReleaseInfoRowWithMaybe(
            CREATED_BY,
            CommonFormat.formatStartedByTrigger(startedByTrigger))
        case startedByCreateReleaseTask: StartedByCreateReleaseTask =>
          addReleaseInfoRowWithMaybe(
            CREATED_BY,
            CommonFormat.formatStartedByMasterRelease(startedByCreateReleaseTask))
      }

      if (isStarted) {
        addReleaseInfoRowWithMaybe("Start date",
          MaybeData(data.releaseTree.root.getItem().asInstanceOf[Release].getStartDate), style = styles.noGridDateOnlyDate)
        addReleaseInfoRowWithMaybe("End date",
          MaybeData.allowingNull(data.releaseTree.root.getItem().asInstanceOf[Release].getEndDate), style = styles.noGridDateOnlyDate)
      }
    }

    val tags = release.getTags.asScala.mkString("; ")
    val maybeTags = Option(tags).filter(_.trim.nonEmpty).getOrElse("")
    val tagsTitle = if (release.isTemplate) {
      TEMPLATE_TAGS
    } else if (release.isWorkflow) {
      WORKFLOW_TAGS
    } else {
      RELEASE_TAGS
    }

    addReleaseInfoRowWithMaybe(tagsTitle, MaybeData(maybeTags))

    addReleaseInfoRowWithMaybe("Flag status",
      MaybeData(Option(release.getRealFlagStatus).map(_.toString.toLowerCase()).map(CommonFormat.prettifyStatusString).orNull))

    val flagComment = release.getFlagComment
    if (flagComment != null && flagComment.nonEmpty) {
      addReleaseInfoRowWithMaybe("", MaybeData(flagComment))
    }

  }

  def addPhaseRow(plannedPhase: PlannerReleaseItem): Unit = {
    sheetWriter.setPhaseRow(true)

    val phase: Phase = plannedPhase.getItem().asInstanceOf[Phase]
    sheetWriter.newRow

    addCellForMaybe(MaybeData.nonEmptyString(CommonFormat.getPlanItemTitle(data.releaseTree.release, phase)))
    sheetWriter
      .addCell("") // task column
      .addCell("") // description column
      .addCell("Phase")
    addCellForMaybe(MaybeData.nonEmptyString(CommonFormat.prettifyStatusString(phase.getStatus.value())))
    addCellForMaybe(MaybeData.allowingNull(phase.getStartDate))
    addCellForMaybe(MaybeData.allowingNull(phase.getEndDate))
    addCellForMaybe(MaybeData.allowingNull(phase.getActualDuration))

    (1 to 3).foreach(_ => sheetWriter.addCell("")) // Empty cell for assignedUser, assignedTeam, CompletedBy columns

    sheetWriter.setPhaseRow(false)
  }

  def addTaskRow(plannedTask: PlannerReleaseItem): Option[(Int, Int)] = {
    sheetWriter.newRow
    val task: Task = plannedTask.getItem().asInstanceOf[Task]
    val taskAgent = data.tasksAgents.getOrElse(task.getId, TaskAgent((Option.empty, Option.empty), Option.empty))
    addTaskCommonCells(plannedTask, data.releaseTree.release, taskAgent, printPhaseTitle = false, includeDescription = true)
  }
}
