package com.xebialabs.xlrelease.reports.excel

import com.xebialabs.xlrelease.reports.domain.MaybeData.Maybe
import com.xebialabs.xlrelease.reports.domain.ReportCellError
import com.xebialabs.xlrelease.reports.excel.ErrorHandler.ERROR_CELL_VALUE
import org.apache.poi.xssf.usermodel.XSSFCellStyle

import scala.util.{Failure, Success}

object ErrorHandler {
  private val TRUNCATED_MESSAGE: String =
    "This value was truncated to fit within one cell. The original length was: "
  private val EXTRA_SPACE: Int = TRUNCATED_MESSAGE.length + 3 + Math.log10(ExcelSheetWriter.MAXIMUM_CELL_LENGTH.toDouble).ceil.toInt
  private val MAX_LENGTH: Int = ExcelSheetWriter.MAXIMUM_CELL_LENGTH - EXTRA_SPACE

  val ERROR_CELL_VALUE = "Error"

  class ValueExceedsMaximumCellSize(original: String) extends RuntimeException(
    s"$TRUNCATED_MESSAGE${original.length()}"
  )
  object ValueExceedsMaximumCellSize {
    def apply(original: String): ValueExceedsMaximumCellSize = new ValueExceedsMaximumCellSize(original)
  }

  def truncate(s: String): Either[(ValueExceedsMaximumCellSize, String), String] = {
    Option(s).collect {
      case _ if s.length > ExcelSheetWriter.MAXIMUM_CELL_LENGTH =>
        val t = ValueExceedsMaximumCellSize(s)
        val m = s.substring(0, MAX_LENGTH) + s"[${t.getMessage}]"
        Left(t -> m)
    }.getOrElse(Right(s))
  }

}

trait ErrorHandler {
  var sheetWriter: ExcelSheetWriter
  val styles: ExcelStyles

  def truncateAndAddCellForMaybe(s: Maybe[Any], style: XSSFCellStyle): ExcelSheetWriter =
    truncateAndAddCellForMaybe(s, Option(style))

  def truncateAndAddCellForMaybe(s: Maybe[Any], style: Option[XSSFCellStyle] = None): ExcelSheetWriter = {
    addMaybe(truncateMaybe(s), style)
  }
  def addCellForMaybe(s: Maybe[Any]): ExcelSheetWriter =
    addMaybe(discardPartialValue(s), None)

  def addCellForMaybe(s: Maybe[Any], style: XSSFCellStyle): ExcelSheetWriter =
    addMaybe(discardPartialValue(s), Option(style))

  private def addMaybe(s: Maybe[Any], style: Option[XSSFCellStyle]) : ExcelSheetWriter = {
    def styleWithErrorFont: XSSFCellStyle = style.fold(styles.error){ s =>
      val customStyle = sheetWriter.getWorkbook.createCellStyle()
      customStyle.cloneStyleFrom(s)
      customStyle.setFont(styles.errorFont)
      customStyle
    }
    s match {
      case Success(Right(v)) =>
        style.fold(sheetWriter.addCell(v))(sheetWriter.addCell(v, _))
      case Success(Left((e, v))) =>
        sheetWriter.getReportWorkbook.addCellError(ReportCellError(sheetWriter, e))
        sheetWriter.addCell(v, styleWithErrorFont)
      case Failure(e) =>
        sheetWriter.getReportWorkbook.addCellError(ReportCellError(sheetWriter, e))
        sheetWriter.addCell(ERROR_CELL_VALUE, styleWithErrorFont)
    }
  }

  private def discardPartialValue(s: Maybe[Any]): Maybe[Any] = s.map {
    case Left((t, _)) => Left(t -> t.getMessage)
    case other => other
  }

  private def truncateMaybe(s: Maybe[Any]): Maybe[Any] =
    discardPartialValue(s).map(_.flatMap {
      case s: String => ErrorHandler.truncate(s)
      case other => Right(other)
    })
}
