package com.xebialabs.xlrelease.reports.upgrade

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.reports.job.domain.ReportJob
import com.xebialabs.xlrelease.reports.job.repository.sql.ReportJobSchema.{REPORT_JOBS => RJ}
import com.xebialabs.xlrelease.reports.job.repository.{ReportJobFilters, ReportJobRepository}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.apache.commons.io.FilenameUtils
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.data.domain.{Page, PageRequest, Sort}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.net.URI
import java.nio.file.Paths
import java.sql.PreparedStatement
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._

@Component
class XLRelease950ReportJobNameUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                                    @Qualifier("xlrRepositoryTransactionTemplate") transactionTemplate: TransactionTemplate,
                                                    reportJobRepository: ReportJobRepository)
  extends Upgrade with Logging {

  private val PAGE_SIZE = 100

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.5.0#4")

  override def doUpgrade(): Boolean = {
    upgradeReportJobs()
    true
  }

  private def upgradeReportJobs(): Unit = {
    upgradePage(0, 0) { page =>
      val rowsToUpdate = page.getContent.asScala.filter(isUpgradeable)
        .map { row =>
          row.setReportName(reportUriToName(row.resultUri))
          row
        }.toList
      transactionTemplate.execute { ts =>
        jdbcTemplate.batchUpdate(
          s"UPDATE ${RJ.TABLE} SET ${RJ.REPORT_NAME} = ? WHERE ${RJ.JOB_ID} = ?",
          new BatchPreparedStatementSetter {
            override def getBatchSize: Int = rowsToUpdate.size

            override def setValues(ps: PreparedStatement, i: Int): Unit = {
              ps.setString(1, rowsToUpdate(i).reportName)
              ps.setInt(2, rowsToUpdate(i).getJobId())
            }
          }
        )
      }
    }
  }

  private def isUpgradeable(job: ReportJob): Boolean = {
    val isAuditReport = job.reportName.toLowerCase() == "XL Release Audit report".toLowerCase()
    val hasResultUri = job.resultUri != null
    isAuditReport && hasResultUri
  }

  private def reportUriToName(uri: String): String = {
    val reportURI = URI.create(uri)
    val path = Paths.get(reportURI.getPath)
    FilenameUtils.removeExtension(path.getFileName.toString)
  }

  @tailrec
  private def upgradePage(pageNumber: Int, rowsUpgraded: Int)(block: Page[ReportJob] => Unit): Int = {
    val pageable = PageRequest.of(pageNumber, PAGE_SIZE, Sort.by("jobId").ascending())
    val filter = new ReportJobFilters()
    val page: Page[ReportJob] = reportJobRepository.query(filter, pageable)

    block(page)

    val upgradedRows = rowsUpgraded + page.getNumberOfElements
    if (!page.hasNext) {
      upgradedRows
    } else {
      upgradePage(pageNumber + 1, upgradedRows)(block)
    }
  }

}
