package com.xebialabs.xlrelease.reports.service

import com.google.common.collect.ImmutableSortedSet
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.api.v1.forms.TimeFrame
import com.xebialabs.xlrelease.reports.api.internal.ReportForm
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.reports.utils.ReportUtils.splitTags

import java.util.{Date, Objects, List => JList}
import scala.jdk.CollectionConverters._

object ReportParams {

  def apply(reportsForm: ReportForm): ReportParams = {
    apply(reportsForm.getTimeFrame, reportsForm.getFrom, reportsForm.getTo,
      splitTags(reportsForm.getTags), reportsForm.getFilters, reportsForm.isUserSpecific, null, reportsForm.isRefresh)
  }

  def apply(timeFrame: TimeFrame, from: Date, to: Date, tags: JList[String], filters: JList[ReportFilter]): ReportParams = {
    apply(timeFrame, from, to, tags, filters, userSpecific = true, null)
  }

  def apply(timeFrame: TimeFrame, from: Date, to: Date, tags: JList[String],
            filters: JList[ReportFilter], userSpecific: Boolean, limit: Integer, refresh: Boolean = false): ReportParams = {
    val ifRange = (d: Date) => if (timeFrame == TimeFrame.RANGE) d else null
    val sortedTags = ImmutableSortedSet.copyOf[String](Option(tags).map(_.asScala.map(_.toLowerCase)).getOrElse(Seq.empty).asJava)
    val username = if (userSpecific) getAuthenticatedUserName else null
    new ReportParams(timeFrame, sortedTags, ifRange(from), ifRange(to), limit, username, userSpecific, filters, refresh)
  }
}

//noinspection ScalaStyle
final class ReportParams(val timeFrame: TimeFrame, val tags: ImmutableSortedSet[String],
                         val from: Date, val to: Date, val limit: Integer,
                         val userName: String, val userSpecific: Boolean, val filters: JList[ReportFilter], val refresh: Boolean = false) {

  override val hashCode: Int = Objects.hash(timeFrame, tags, from, to, limit, userName, Option(filters).map(_.asScala.map(_.toString.hashCode)))

  override def equals(other: Any): Boolean = other match {
    case that: ReportParams =>
      if (other.hashCode != hashCode) false
      else
        timeFrame == that.timeFrame &&
          tags == that.tags &&
          from == that.from &&
          to == that.to &&
          limit == that.limit &&
          userName == that.userName &&
          Objects.equals(filters, that.filters)
    case _ => false
  }
}
