package com.xebialabs.xlrelease.reports.job.impl

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.metrics.MetricsConfiguration.TAG_EXECUTOR
import com.xebialabs.xlrelease.reports.job.impl.ReportExecutorService._
import com.xebialabs.xlrelease.utils.PrefixedThreadFactory
import grizzled.slf4j.Logging
import io.micrometer.core.instrument.MeterRegistry
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics

import java.util.concurrent.{Callable, Executors, Future, TimeUnit}
import scala.concurrent.{ExecutionContext, ExecutionContextExecutorService}

class ReportExecutorService(xlrConfig: XlrConfig, meterRegistry: MeterRegistry) extends Logging {

  private val executorService = {
    val executor = Executors.newFixedThreadPool(xlrConfig.reporting.engine.maxThreadsCount, new PrefixedThreadFactory(REPORT_POOL_NAME, true))
    if (xlrConfig.metrics.enabled) {
      ExecutorServiceMetrics.monitor(meterRegistry, executor, "report-executor", TAG_EXECUTOR)
    } else {
      executor
    }
  }

  lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(executorService)

  def shutdown(): Unit = {
    logger.info(s"Shutting down thread pool '$REPORT_POOL_NAME'")
    executorService.shutdown()
    try {
      val timeout = 60
      if (!executorService.awaitTermination(timeout, TimeUnit.SECONDS)) {
        executorService.shutdownNow()
        if (!executorService.awaitTermination(timeout, TimeUnit.SECONDS)) {
          logger.error(s"Unable to shutdown thread pool '$REPORT_POOL_NAME' in ${2 * timeout} seconds")
        }
      }
      logger.info(s"Shut down thread pool '$REPORT_POOL_NAME'")
    } catch {
      case _: InterruptedException =>
        logger.info(s"Interrupted while shutting down thread pool '$REPORT_POOL_NAME'")
        executorService.shutdownNow()
        Thread.currentThread().interrupt()
    }
  }

  def submit[T](task: Callable[T]): Future[T] = executorService.submit(task)
}

object ReportExecutorService {
  final val REPORT_POOL_NAME = "report"
}
