package com.xebialabs.xlrelease.reports.excel

import com.xebialabs.xlrelease.reports.audit.InstanceData
import org.apache.poi.ss.util.CellRangeAddress
import org.apache.poi.xssf.usermodel._

import java.util.Date

trait ExcelCommonHeader {

  val HEADER_TITLE = "Digital.ai Release Audit Report"
  val TEMPLATE_HEADER_TITLE = "Template Audit Report"
  val RELEASE_INSTANCE = "Release instance"

  def sheetWriter: ExcelSheetWriter
  def workbook: ReportWorkbook
  def styles: ExcelStyles

  case class ReportInfo(reportTitle: String = HEADER_TITLE, masterReportName: String = "", infoLabel: String = "Report info")

  def addCommonHeader(username: String,
                      creationDate: Date,
                      instanceData: InstanceData,
                      cellsInRow: Integer,
                      logoWidth: Int,
                      logoHeight: Int,
                      reportInfo: ReportInfo): Unit = {

    addWhiteOnGreenTitleBar(sheetWriter, reportInfo.reportTitle, cellsInRow)
    addNoBorderEmptyRow(cellsInRow)
    addReportInfoCommonBlock(reportInfo, username, creationDate, instanceData, cellsInRow)

    if (reportInfo.masterReportName != null && !reportInfo.masterReportName.isEmpty) {
      sheetWriter
        .newRow()
        .addEmptyCell(styles.noGrid)
        .addCell("Master report", styles.leftAlignedNoGrid)
        .addCell(reportInfo.masterReportName, styles.noGrid)
        .mergeCells(
          new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex))
      (1 to 3).foreach(_=>sheetWriter.addEmptyCell(styles.noGrid))
    } else {
      sheetWriter
        .newRow()
      (1 to 6).foreach(_=>sheetWriter.addEmptyCell(styles.noGrid))
    }

    //Version under logo
    sheetWriter.addCell(instanceData.version.getOrElse(""), styles.rightAlignedNoGrid)
    fillCellsUntilEndOfRow(cellsInRow, styles.noGrid)
    addXlLogo(sheetWriter, logoWidth, logoHeight)
  }

  private def addReportInfoCommonBlock(reportInfo: ReportInfo, username: String, creationDate: Date, instanceData: InstanceData, cellsInRow: Integer) = {
    sheetWriter
      .newRow()
      .addCell(reportInfo.infoLabel, styles.noGridBoldText)
      .addCell("Generated on", styles.leftAlignedNoGrid)
      .addCell(creationDate, styles.noGridDate)
      .mergeCells(
        new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex))
    fillCellsUntilEndOfRow(cellsInRow, styles.noGrid)

    sheetWriter
      .newRow()
      .addEmptyCell(styles.noGrid)
      .addCell("Generated by", styles.leftAlignedNoGrid)
      .addCell(username, styles.noGrid)
      .mergeCells(
        new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex))
    fillCellsUntilEndOfRow(cellsInRow, styles.noGrid)

    sheetWriter
      .newRow()
      .addEmptyCell(styles.noGrid)
      .addCell(RELEASE_INSTANCE, styles.leftAlignedNoGrid)

    instanceData.url.fold {
      sheetWriter.addEmptyCell(styles.noGrid)
    } {
      url => sheetWriter.addHyperlink(instanceData.name.getOrElse(url), url, styles.hyperlink)
    }

    sheetWriter.mergeCells(
      new CellRangeAddress(sheetWriter.rowIndex - 1, sheetWriter.rowIndex - 1, sheetWriter.columnIndex - 1, sheetWriter.columnIndex)
    )

    fillCellsUntilEndOfRow(cellsInRow, styles.noGrid)
  }

  protected def addNoBorderEmptyRow(cellsInRow: Integer): Unit = {
    sheetWriter.newRow()
    fillCellsUntilEndOfRow(cellsInRow, styles.noGrid)
  }

  private def addWhiteOnGreenTitleBar(sheetWriter: ExcelSheetWriter, reportTitle: String, cellsInRow: Integer): Unit = {
    sheetWriter
      .newRow()
      .setRowHeight(22)
      .addCell(reportTitle, styles.whiteOnGreenBig)

    fillCellsUntilEndOfRow(cellsInRow, styles.whiteOnGreen)
  }

  def addXlLogo(sheetWriter: ExcelSheetWriter, width: Int, height: Int): Unit = {
    val imageData = this.getClass.getResource("/digital-ai-release-logo.png")
    sheetWriter.insertImage(imageData, 1, 5, width, height)
  }

  protected def fillCellsUntilEndOfRow(cellsInRow: Int, style: XSSFCellStyle): Unit = {
    while (sheetWriter.getColumnIndex < cellsInRow) {
      sheetWriter.addEmptyCell(style)
    }
  }

}
